package org.nsh07.wikireader.ui.image

import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.material3.CircularWavyProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3ExpressiveApi
import androidx.compose.material3.LoadingIndicator
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import coil3.compose.AsyncImagePainter
import coil3.compose.rememberAsyncImagePainter
import coil3.request.ImageRequest
import coil3.request.crossfade

@OptIn(ExperimentalMaterial3ExpressiveApi::class)
@Composable
fun FeedImage(
    source: String?,
    modifier: Modifier = Modifier,
    description: String? = null,
    width: Int? = null,
    height: Int? = null,
    loadingIndicator: Boolean,
    background: Boolean,
    colorFilter: ColorFilter? = null,
    contentScale: ContentScale = ContentScale.Crop
) {
    val context = LocalContext.current
    val painter = rememberAsyncImagePainter(
        model = ImageRequest.Builder(context)
            .data(source)
            .crossfade(true)
            .build(),
        contentScale = contentScale,
    )

    val painterState by painter.state.collectAsStateWithLifecycle()

    when (painterState) {
        is AsyncImagePainter.State.Success -> {
            Image(
                painter = painter,
                contentDescription = description,
                contentScale = contentScale,
                colorFilter = colorFilter,
                modifier =
                    if (width != null && height != null)
                        modifier
                            .fillMaxWidth()
                            .aspectRatio(width.toFloat() / height.toFloat())
                            .background(if (background) Color.White else Color.Transparent)
                    else
                        modifier
                            .fillMaxSize()
                            .background(if (background) Color.White else Color.Transparent)
            )
        }

        is AsyncImagePainter.State.Loading -> {
            Box(
                contentAlignment = Alignment.Center,
                modifier =
                    if (width != null && height != null)
                        modifier
                            .fillMaxWidth()
                            .aspectRatio(width.toFloat() / height.toFloat())
                    else
                        modifier
                            .fillMaxSize()
            ) {
                if (loadingIndicator) LoadingIndicator()
                else CircularWavyProgressIndicator()
            }
        }

        else -> {
            Spacer(
                modifier =
                    if (width != null && height != null)
                        modifier
                            .fillMaxWidth()
                            .aspectRatio(width.toFloat() / height.toFloat())
                    else
                        modifier.fillMaxSize()
            )
        }
    }
}