/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.richertext

import org.grating.styncynotes.ui.data.StickyPyFormatterTest.Companion.SAMPLE_STICKY_PY
import org.grating.styncynotes.ui.data.fromStickyPy
import org.grating.styncynotes.ui.data.toStickyPy
import org.junit.Assert.assertEquals

import org.junit.Test

@Suppress("SameParameterValue")
class StickyStringTest {

    @Test
    fun applyFormat() {
        var sticky = fromStickyPy(SAMPLE_STICKY_PY)
        val text = sticky.getText()

        // Bold at beginning.
        sticky.applyFormat(Format.BOLD, 0, 10)
        var actual = toStickyPy(sticky)
        var expected = "#tag:bold:An example#tag:bold: note to #tag:monospace:demonstrate"
        assertEquals(expected, actual.substring(0, expected.length))

        // Highlight at end.
        sticky = fromStickyPy(SAMPLE_STICKY_PY)
        val textToHighlight = "Large Text\n" +
                "Larger Text\n" +
                "Header Text\n" +
                "asdfasdfasdf\n"
        sticky.applyFormat(Format.HIGHLIGHT, text.length - textToHighlight.length, text.length)
        actual = toStickyPy(sticky)
        expected = "Normal Text\n" +
                "#tag:small:Small Text\n" +
                "#tag:highlight:#tag:small:#tag:large:Large Text\n" +
                "#tag:large:#tag:larger:Larger Text\n" +
                "#tag:header:Hea#tag:larger:#tag:small:der T#tag:small:#tag:larger:ext#tag:header:\n" +
                "asdfasdfasdf\n" +
                "#tag:highlight:#tag:larger:"
        assertEquals(expected, actual.substring(actual.length - expected.length, actual.length))
    }

    @Test
    fun insertText() {
        val stickyPy =
            "An example note to #tag:monospace:demonstrate#tag:monospace: the different rendering " +
                    "functionality of StickyNotes.py and Gtk."

        // Insert at start.
        insertTest(
            startPy = stickyPy,
            insertThis = "-Text at start-",
            insertAt = 0,
            expectedPy = "-Text at start-An example note to #tag:monospace:demonstrate#tag:monospace:" +
                    " the different rendering functionality of StickyNotes.py and Gtk.")

        // Insert immediately before a style starts.
        insertTest(
            startPy = stickyPy,
            insertThis = "-Should push monospace right-",
            insertAt = 19,
            expectedPy = "An example note to -Should push monospace right-#tag:monospace:demonstrate#tag:monospace:" +
                    " the different rendering functionality of StickyNotes.py and Gtk.")

        // Insert in the middle of a span.
        insertTest(
            startPy = stickyPy,
            insertThis = "-Should split demonstrate-",
            insertAt = 24,
            expectedPy = "An example note to #tag:monospace:demon-Should split demonstrate-strate#tag:monospace: the" +
                    " different rendering functionality of StickyNotes.py and Gtk.")

        // Insert at end of a span.
        insertTest(
            startPy = stickyPy,
            insertThis = "-Should run on after demonstrate-",
            insertAt = 30,
            expectedPy = "An example note to #tag:monospace:demonstrate-Should run on after demonstrate-#tag:monospace: " +
                    "the different rendering functionality of StickyNotes.py and Gtk.")

        // Insert at end of text.
        insertTest(
            startPy = stickyPy,
            insertThis = "-Should run on after Gtk-",
            insertAt = fromStickyPy(stickyPy).getText().length,
            expectedPy = "An example note to #tag:monospace:demonstrate#tag:monospace: the different rendering " +
                    "functionality of StickyNotes.py and Gtk.-Should run on after Gtk-")
    }

    @Test
    fun deleteText() {
        val stickyPy =
            "An example note to #tag:monospace:demonstrate#tag:monospace: the different rendering " +
                    "functionality of StickyNotes.py and Gtk."
        val txtLen = fromStickyPy(stickyPy).getText().length

        // Delete nothing at start position.
        deleteTest(
            startPy = stickyPy,
            startPos = 0,
            endPos = 0,
            expectedPy = stickyPy)

        // Delete nothing at end position.
        deleteTest(
            startPy = stickyPy,
            startPos = txtLen - 1,
            endPos = txtLen - 1,
            expectedPy = stickyPy)

        // Delete first char.
        deleteTest(
            startPy = stickyPy,
            startPos = 0,
            endPos = 1,
            expectedPy = "n example note to #tag:monospace:demonstrate#tag:monospace: the different rendering " +
                    "functionality of StickyNotes.py and Gtk.")

        // Delete all chars in first span.
        deleteTest(
            startPy = stickyPy,
            startPos = 0,
            endPos = 19,
            expectedPy = "#tag:monospace:demonstrate#tag:monospace: the different rendering " +
                    "functionality of StickyNotes.py and Gtk.")

        // Delete all chars in first span and some in second.
        deleteTest(
            startPy = stickyPy,
            startPos = 0,
            endPos = 24,
            expectedPy = "#tag:monospace:strate#tag:monospace: the different rendering " +
                    "functionality of StickyNotes.py and Gtk.")

        // Delete all chars.
        deleteTest(
            startPy = stickyPy,
            startPos = 0,
            endPos = txtLen,
            expectedPy = "")

        // Delete chars halfway through first span to a few chars into second.
        deleteTest(
            startPy = stickyPy,
            startPos = 10,
            endPos = 24,
            expectedPy = "An example#tag:monospace:strate#tag:monospace: the different rendering " +
                    "functionality of StickyNotes.py and Gtk.")

        // Delete all of middle span.
        deleteTest(
            startPy = stickyPy,
            startPos = 19,
            endPos = 30,
            expectedPy = "An example note to  the different rendering functionality of StickyNotes.py and Gtk.")

        // Delete all of last span.
        deleteTest(
            startPy = stickyPy,
            startPos = 30,
            endPos = txtLen,
            expectedPy = "An example note to #tag:monospace:demonstrate")

        // Delete end of middle and start of last span.
        deleteTest(
            startPy = stickyPy,
            startPos = 25,
            endPos = 44,
            expectedPy = "An example note to #tag:monospace:demons#tag:monospace: rendering " +
                    "functionality of StickyNotes.py and Gtk.")

        // Delete chunk last span.
        deleteTest(
            startPy = stickyPy,
            startPos = 35,
            endPos = 44,
            expectedPy = "An example note to #tag:monospace:demonstrate#tag:monospace: the  rendering " +
                    "functionality of StickyNotes.py and Gtk.")
    }

    /**
     * Convenience function for testing text inserts, primarily to make testing code more readable.
     */
    private fun insertTest(
        startPy: String,
        insertThis: String,
        insertAt: Int,
        expectedPy: String
    ) {
        val sticky = fromStickyPy(startPy)
        sticky.insertText(insertThis, insertAt)
        val actual = toStickyPy(sticky)
        assertEquals(expectedPy, actual)
    }

    private fun deleteTest(
        startPy: String,
        startPos: Int,
        endPos: Int,
        expectedPy: String
    ) {
        val sticky = fromStickyPy(startPy)
        sticky.deleteText(startPos, endPos)
        val actual = toStickyPy(sticky)
        assertEquals(expectedPy, actual)
    }
}