/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.richertext

import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.AnnotatedString.Range
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextRange
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.unit.sp
import org.grating.styncynotes.ifNeg
import org.grating.styncynotes.logInfo2
import org.grating.styncynotes.showSpanStyleDeltasAlongsideText
import kotlin.math.max

val DEFAULT_INITIAL_SPAN_STYLE = SpanStyle(fontSize = 14.sp)

/**
 * AnnotatedString "decorator" (not strictly, but in essence) that adds a flags list alongside the
 * text / spans / annotations etc.
 */
class StickyString(initialStyle: SpanStyle = DEFAULT_INITIAL_SPAN_STYLE) {
    private val text = StringBuilder()
    private var dirty: Boolean = false

    private val textLength: Int
        get() = text.length

    private val initialStyleRange = Range(initialStyle, 0, 0)
    private val initialFlagsRange = Range(Format.EMPTY_FLAGS, 0, 0)


    // NOTE: styles and flagss are two parallel lists (always the same length with the same ranges)
    // Not condense into a single list of pairs for ease and efficiency when generating annotated-
    // strings (which only know nothing about flags).
    private val styles = mutableListOf(initialStyleRange)
    private val flagss = mutableListOf(initialFlagsRange)

    @Synchronized
    fun getFlags() = flagss.toList()

    @Synchronized
    fun getText() = text.toString()

    @Synchronized
    fun getPreviousLine(pos: Int): String =
        text.substring(text.lastIndexOf('\n', max(pos - 1, 0)).let { if (it == -1) 0 else it + 1 },
                       pos) // off by 1 hell

    /**
     * Get an annotated string that can be used in GUI widgets.
     */
    val annotatedString: AnnotatedString
        @Synchronized
        get() = AnnotatedString(text.toString(), styles.toList())

    /**
     * Push txt to the end of the current text.  Increment the end ranges of the styles and flags
     * to accommodate.
     *
     * @param txt the text to append.
     */
    @Synchronized
    fun push(txt: String) {
        dirty = true
        text.append(txt)
        styles[styles.lastIndex] = styles.last().copy(end = text.length)
        flagss[flagss.lastIndex] = flagss.last().copy(end = text.length)
    }

    /**
     * Push the format so that it is toggled for any subsequent pushed text.
     * @param format the format to make active (or inactive if it is already active).
     */
    @Synchronized
    fun push(format: Format) {
        dirty = true
        with(format.toggle(styles.last().item, flagss.last().item)) {
            styles.add(Range(first, text.length, text.length))
            flagss.add(Range(second, text.length, text.length))
        }
    }

    private fun getSpan(idx: Int): Pair<Range<SpanStyle>, Range<String>> =
        Pair(styles[idx], flagss[idx])

    private fun setSpan(idx: Int, style: Range<SpanStyle>, flags: Range<String>) {
        styles[idx] = style; flagss[idx] = flags
    }

    private fun shiftLeft(idx: Int, amount: Int) {
        styles[idx] = styles[idx].let { it.copy(start = it.start - amount, end = it.end - amount) }
        flagss[idx] = flagss[idx].let { it.copy(start = it.start - amount, end = it.end - amount) }
    }

    private fun delete(idx: Int) {
        styles.removeAt(idx)
        flagss.removeAt(idx)
    }

    private fun addSpan(idx: Int, style: Range<SpanStyle>, flags: Range<String>) {
        styles.add(idx, style); flagss.add(idx, flags)
    }

    /**
     * Apply the format to the range supplied.  If the format is currently off AT THE START OF THE
     * RANGE, toggle it on, else toggle it off.
     * @param format the format to apply.
     * @param cPos1 the first cursor position (inclusive) to apply the format.
     * @param cPos2 the last cursor position (inclusive) to apply the format.
     */
    @Synchronized
    fun applyFormat(format: Format, cPos1: Int, cPos2: Int) {
        logInfo2("BEFORE: " + showSpanStyleDeltasAlongsideText(annotatedString))

        dirty = true
        val (start, end) = listOf(cPos1, cPos2).sorted()
        val (startSpanIdx, endSpanIdx) = splitSpansToBoundaries(start, end)

        // Now we have a range (startSpanIdx -> endSpanIdx) of whole existing spans that are effected,
        // so merge the new style into all of them.
        val isOn: Boolean = format.state(flagss[startSpanIdx].item) == Format.State.ON
        for (idx in startSpanIdx..endSpanIdx) {
            val (newSpan, newFlags) = if (isOn)
                format.disable(styles[idx].item, flagss[idx].item)
            else
                format.enable(styles[idx].item, flagss[idx].item)
            setSpan(idx, styles[idx].copy(item = newSpan), flagss[idx].copy(item = newFlags))
        }
        logInfo2("AFTER: " + showSpanStyleDeltasAlongsideText(annotatedString))
    }

    /**
     * Insert the supplied text at the specified position (inheriting the prevailing style present
     * at this position).
     * @param txt The text to insert.
     * @param pos where to insert the text.
     */
    @Synchronized
    fun insertText(txt: String, pos: Int): StickyString {
        dirty = true
        text.insert(pos, txt)
        return adjustSpansForInsert(pos, txt.length)
    }

    /**
     * Delete the text that lies between start (inclusive) and end (exclusive).
     * @param start the first character (inclusive) in the range to delete.
     * @param end the last character (exclusive) in the range to delete.
     */
    @Synchronized
    fun deleteText(start: Int, end: Int): StickyString {
        return if (start == end)
            this
        else {
            dirty = true
            // checkCoverage()
            text.delete(start, end) // End can be greater than length!
            adjustSpansForDelete(start, end)
        }
    }

    /**
     * Replace the string that lies at <code>pos</code> with the specified string.
     * Because the replacement is the same length as the string being replaced, we need not do
     * anything with the spans.
     */
    @Synchronized
    fun replaceText(pos: Int, txt: String): StickyString {
        dirty = true
        text.replace(pos, pos + txt.length, txt)
        return this
    }

    /**
     * Adjust spans following a text insert.
     */
    private fun adjustSpansForInsert(pos: Int, amount: Int): StickyString {
        // Find the style that was in effect where chars were inserted.
        var idx = styles.binarySearch { it.end - pos }.ifNeg { -it - 1 }
            .coerceAtMost(styles.lastIndex)
        while (idx + 1 < styles.size && styles[idx + 1].end == pos) idx++ // Accommodate a newly added style.

        // Expand span of style where insert occurred.
        styles[idx] = styles[idx].let { it.copy(end = it.end + amount) }
        flagss[idx] = flagss[idx].let { it.copy(end = it.end + amount) }

        // Adjust styles fully to the right of the change.
        while (++idx < styles.size) {
            styles[idx] =
                styles[idx].let { it.copy(start = it.start + amount, end = it.end + amount) }
            flagss[idx] =
                flagss[idx].let { it.copy(start = it.start + amount, end = it.end + amount) }
        }
        return this
    }

    /**
     * Adjust spans following a text delete.
     */
    private fun adjustSpansForDelete(cursor1: Int, cursor2: Int): StickyString {
        dirty = true
        val (start, end) = listOf(cursor1, cursor2).sorted()
        var (startSpanIdx, endSpanIdx) = splitSpansToBoundaries(start, end)

        // Delete the spans that catered for the deleted text.
        while (endSpanIdx >= startSpanIdx)
            delete(endSpanIdx--)

        // Shrink the remaining spans.
        val shift = end - start
        while (++endSpanIdx < styles.size)
            shiftLeft(endSpanIdx, shift)

        // HACK-YUK: If text was completely deleted and we removed all styles and flags, then re-init.
        if (textLength == 0 && styles.size == 0) {
            styles.add(initialStyleRange)
            flagss.add(initialFlagsRange)
        }
        return this
    }

    /**
     * Every character of text should be covered by a style.  If this is no longer the case then
     * blow up.
     * NOTE: This function is handy for testing/debugging, but otherwise isn't used.  So I'm leaving
     * it in.
     */
    @Suppress("unused")
    private fun checkCoverage() {
        var start = -1
        var end = -1
        fun yuk(idx: Int) {
            throw IllegalStateException("Gap in styles @ $idx:\n" + styles.joinToString("\n"))
        }
        styles.forEachIndexed { index, range ->
            if (start == -1) {
                if (range.start != 0) yuk(0)
                start = range.start
                end = range.end
            } else {
                if (range.start != end) yuk(index)
                start = range.start; end = range.end
            }
        }
        if (end != text.length) yuk(end)
        logInfo2("Text Length: ${text.length} Span End: $end")
    }

    @Synchronized
    fun isDirty() = dirty

    @Synchronized
    fun resetDirty() {
        dirty = false
    }

    /**
     * Split spans so that start is the start of a span and end is the end of a span.  When span
     * boundaries lie on operation boundaries (e.g. format, delete) the logic becomes a lot
     * simpler.
     *
     * @param start start of range on which a span boundary should occur.
     * @param end end of range on which a span boundary should occur.
     */
    private fun splitSpansToBoundaries(start: Int, end: Int): Pair<Int, Int> {
        check(start <= end) { throw IllegalArgumentException("Negative range [$start,$end] not supported.") }

        // Find the indexes of the span that encompasses 'start' and the span that encompasses 'end'
        // (which may be the same span).
        // -2 because we should split the span within which the boundary starts (-1 gives the insertion point which would be the next span)
        var startSpanIdx = styles.binarySearch { it.start - start }.ifNeg { -it - 2 }
        var endSpanIdx = styles.binarySearch(startSpanIdx) { it.end - end }.ifNeg { -it - 1 }
            .coerceAtMost(styles.lastIndex)

        // If the start span straddles 'start' then split it.
        val (style, flags) = getSpan(startSpanIdx)
        if (start > style.start) {
            setSpan(startSpanIdx, style.copy(end = start), flags.copy(end = start))
            addSpan(startSpanIdx + 1,
                    style.copy(start = start, end = style.end),
                    flags.copy(start = start, end = flags.end))
//            logInfo2("Split start [${style.start},${style.end}) -> " +
//                             "[${styles[startSpanIdx].start},${styles[startSpanIdx].end})," +
//                             "[${styles[startSpanIdx + 1].start},${styles[startSpanIdx + 1].end}),")
            startSpanIdx++
            endSpanIdx++
        } else {
//            logInfo2("No split [${style.start},${style.end})")
        }

        // If the end span is straddles 'end' then split that too.
        val (endStyle, endFlags) = getSpan(endSpanIdx)
        if (end < endStyle.end) {
            setSpan(endSpanIdx, endStyle.copy(end = end), endFlags.copy(end = end))
            addSpan(endSpanIdx + 1,
                    endStyle.copy(start = end, end = endStyle.end),
                    endFlags.copy(start = end, end = endFlags.end))
//            logInfo2("Split end [${style.start},${style.end}) -> " +
//                             "[${styles[endSpanIdx].start},${styles[endSpanIdx].end})," +
//                             "[${styles[endSpanIdx + 1].start},${styles[endSpanIdx + 1].end}),")

        }

        return Pair(startSpanIdx, endSpanIdx)
    }
}

fun TextFieldValue.copy(
    stickyString: StickyString,
    selection: TextRange = this.selection
): TextFieldValue =
    copy(annotatedString = stickyString.annotatedString, selection = selection)

