/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.richertext

import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.relocation.BringIntoViewRequester
import androidx.compose.foundation.relocation.bringIntoViewRequester
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.TextFieldColors
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Color.Companion.Transparent
import androidx.compose.ui.text.TextLayoutResult
import androidx.compose.ui.text.input.TextFieldValue
import org.grating.styncynotes.data.Note
import org.grating.styncynotes.highlight
import org.grating.styncynotes.logInfo
import org.grating.styncynotes.ui.model.StyncyNotesViewModel
import org.grating.styncynotes.ui.richertext.RichTextEvent.CheckboxTap
import org.grating.styncynotes.ui.richertext.RichTextEvent.CompositionChange
import org.grating.styncynotes.ui.richertext.RichTextEvent.CursorMove
import org.grating.styncynotes.ui.richertext.RichTextEvent.Nothing
import org.grating.styncynotes.ui.richertext.RichTextEvent.SelectionChange
import org.grating.styncynotes.ui.richertext.RichTextEvent.TypingEvent
import org.grating.styncynotes.ui.simpleVerticalScrollbar
import org.grating.styncynotes.withSpanStylesFrom

private const val LOG_TAG = "RicherTextPane"

/**
 * Display and edit formatted text like the Stick.py app.
 */
@OptIn(ExperimentalFoundationApi::class)
@Composable
fun RicherTextPane(
    viewModel: StyncyNotesViewModel,
    note: Note,
    foreground: Color,
    background: Color,
    scrollable: Boolean,
    onSelected: () -> Unit,
    onValueChange: (TextFieldValue) -> Unit
) {
    val currTfv = note.textFieldValue
    val stickyString = note.stickyString

    val scrollState = rememberScrollState()
    val bringIntoViewRequester by remember { mutableStateOf(BringIntoViewRequester()) }
    var textLayoutResult by remember { mutableStateOf<TextLayoutResult?>(null)}

    LaunchedEffect(currTfv.selection.start, scrollState.viewportSize) {
        textLayoutResult?.let {
            val cursorRect = it.getCursorRect(currTfv.selection.start)
            bringIntoViewRequester.bringIntoView(cursorRect.inflate(cursorRect.height))
        }
    }

    // Test: If we ever get two AddText events in quick succession that are identical, this is a problem
    // with onValueChange.  Ignore the second one.
    var lastRte: RichTextEvent by remember { mutableStateOf(Nothing) }
    MonkeyTextField(value = currTfv,
                    onValueChange = { newTfv ->
                        val rte = whatHappened(currTfv, newTfv)
                        when (rte) {
                            Nothing -> logInfo(LOG_TAG, "!!! $rte !!!")
                            lastRte -> logInfo(LOG_TAG, "Filtering bogus duplicate event: $rte")
                            else -> {
                                logInfo(LOG_TAG, rte)
                                onValueChange(
                                    when (rte) {
                                        is TypingEvent -> accommodateTyping(newTfv,
                                                                            stickyString,
                                                                            rte)

                                        is CheckboxTap -> toggleCheckBox(newTfv, stickyString, rte)
                                        is CursorMove -> newTfv.withSpanStylesFrom(currTfv)
                                        is SelectionChange -> newTfv.withSpanStylesFrom(currTfv)
                                        is CompositionChange -> newTfv.withSpanStylesFrom(currTfv)
                                        Nothing -> throw IllegalStateException("Should never get here.")
                                    })
                            }
                        }
                        lastRte = rte
                    },
                    onTextLayout = {
                        textLayoutResult = it
                    },
                    colors = getTextFieldColors(foreground, background),
                    modifier = Modifier
                        .fillMaxWidth()
                        .simpleVerticalScrollbar(scrollState)
                        .verticalScroll(state = scrollState, enabled = scrollable)
                        .focusRequester(viewModel.rtFocusRequester)
                        .onFocusChanged {
                            if (it.isFocused)
                                onSelected()
                        }
                        .bringIntoViewRequester(bringIntoViewRequester)
    )
}

infix fun <T1, T2> T1.and(other: T2): Pair<T1, T2> = Pair(this, other)

@Composable
fun getTextFieldColors(
    foreground: Color,
    background: Color
): TextFieldColors {
    return TextFieldDefaults.colors(
        focusedTextColor = foreground,
        unfocusedTextColor = foreground,
        focusedContainerColor = background.highlight(Color.White),
        unfocusedContainerColor = background,
        disabledContainerColor = background.highlight(Color.Black),
        focusedIndicatorColor = Transparent,
        unfocusedIndicatorColor = Transparent,
    )
}

