/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes

import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.AnnotatedString.Range
import androidx.compose.ui.text.ParagraphStyle
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.sp

/**
 * Class for utility functions that help with GUI text manipulation leaning towards the Sticky.py
 * format.
 */

fun AnnotatedString.copy(
    text: String = this.text,
    spanStyles: List<Range<SpanStyle>> = this.spanStyles,
    paragraphStyles: List<Range<ParagraphStyle>> = this.paragraphStyles
): AnnotatedString =
    AnnotatedString(text = text, spanStyles = spanStyles, paragraphStyles = paragraphStyles)

fun TextFieldValue.withSpanStylesFrom(other: TextFieldValue) =
    copy(annotatedString = annotatedString.copy(spanStyles = other.annotatedString.spanStyles))


/**
 * Convert the annotated string to a string containing the original text, interspersed with the
 * span styles in the annotated string, output at the locations where the changes to style take
 * effect.
 *
 * NOTE: We assume NO OVERLAPPING SPAN STYLES AND SPAN STYLES BEING IN ORDER.
 *
 * @param annotatedString the source annotated string to be output.
 * @return The text of the annotated string, interspersed with changes to to span style, output
 * at the point where the changes take effect.
 */
fun showSpanStyleDeltasAlongsideText(annotatedString: AnnotatedString): String {
    val styles = annotatedString.spanStyles.iterator()
    val none = SpanStyle()
    var active = Range(none, -1, 0)
    var pending: Range<SpanStyle>? = styles.next()

    val result = StringBuilder()
    annotatedString.text.forEachIndexed { index, c ->
        while (index >= active.end) {
            when {
                pending == null ->
                    result.append("SpanStyles Exhausted! ${SpanStyle().replaces(active.item)}")

                pending!!.start == active.end -> {
                    result.append("<${pending!!.item.replaces(active.item)}>")
                    active = pending!!
                }

                else -> result.append("SpanStyles GAP! [${active.end} .. ${pending!!.start}] " +
                                              pending!!.item.replaces(active.item))
            }
            pending = if (styles.hasNext()) styles.next() else null
        }
        result.append(c)
    }

    // Render any trailing tags.
    while (pending != null) {
        if (pending!!.start != pending!!.end || pending!!.end != annotatedString.text.length)
            throw IllegalStateException("Trailing span style range falls outside text bounds.")
        result.append("<${pending!!.item.replaces(active.item)}>")
        active = pending!!
        pending = if (styles.hasNext()) styles.next() else null
    }
    return result.toString()
}

infix fun TextDecoration.with(other: TextDecoration?): TextDecoration =
    plus(other ?: TextDecoration.None)

infix fun TextDecoration.without(other: TextDecoration?): TextDecoration =
    when {
        other == null || other == TextDecoration.None || !contains(other) -> this
        this == other -> TextDecoration.None
        other == TextDecoration.Underline -> TextDecoration.LineThrough
        else -> TextDecoration.Underline
    }

/**
 * Show differences between two span styles in a "useful" way for debugging etc.
 */
private fun SpanStyle.replaces(other: SpanStyle): String {
    val bldr = StringBuilder()
    bldr.append("SpanStyle(")
    bldr.append("${other.fontSize}->$fontSize".unless { fontSize eq other.fontSize })
    bldr.append("${other.fontWeight}->$fontWeight".unless { fontWeight == other.fontWeight })
    bldr.append("${other.fontStyle}->$fontStyle".unless { fontStyle == other.fontStyle })
    bldr.append("${other.fontSynthesis}->$fontSynthesis".unless { fontSynthesis == other.fontSynthesis })
    bldr.append("${other.fontFamily}->$fontFamily".unless { fontFamily == other.fontFamily })
    bldr.append("${other.fontFeatureSettings}->$fontFeatureSettings".unless { fontFeatureSettings == other.fontFeatureSettings })
    bldr.append("${other.letterSpacing}->$letterSpacing".unless { letterSpacing == other.letterSpacing })
    bldr.append("${other.baselineShift}->$baselineShift".unless { baselineShift == other.baselineShift })
    bldr.append("${other.textGeometricTransform}->$textGeometricTransform".unless { textGeometricTransform == other.textGeometricTransform })
    bldr.append("${other.localeList}->$localeList".unless { localeList == other.localeList })
    bldr.append("${other.background}->$background".unless { background == other.background })
    bldr.append("${other.textDecoration}->$textDecoration".unless { textDecoration == other.textDecoration })
    bldr.append("${other.shadow}->$shadow".unless { shadow == other.shadow })
    bldr.append("${other.platformStyle}->$platformStyle".unless { platformStyle == other.platformStyle })
    bldr.append("${other.drawStyle}->$drawStyle".unless { drawStyle == other.drawStyle })
    bldr.append(")")
    if (bldr.toString() == "SpanStyle()")
        logInfo("", "STOP")
    return bldr.toString()
}

private fun String.unless(test: () -> Boolean): String {
    return if (test()) "" else this
}

infix fun TextUnit.or(other: TextUnit): TextUnit = when {
    this == TextUnit.Unspecified -> other
    else -> this
}

infix fun TextUnit.eq(other: TextUnit): Boolean =
    this.value equalsIsh other.value

val DEFAULT_FONT_SIZE = 14.sp
fun TextUnit.scaleBy(scalar: Double): TextUnit {
    check(isSp) { throw IllegalStateException("I assumed you'd be sizing in sp.") }
    return if (this == TextUnit.Unspecified) {
        (DEFAULT_FONT_SIZE.value * scalar).toNdp(2).sp
    } else {
        (value * scalar).toNdp(2).sp
    }
}

val <T> Range<T>.width: Int
    get():Int = end - start
