/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.data

import androidx.compose.ui.text.SpanStyle
import org.grating.styncynotes.ui.richertext.DEFAULT_INITIAL_SPAN_STYLE
import org.grating.styncynotes.ui.richertext.Format.Companion.FORMATS_BY_TAG
import org.grating.styncynotes.ui.richertext.StickyString

const val CHECKBOX_OFF_TAG = "#check:0"
const val CHECKBOX_ON_TAG = "#check:1"
const val BULLET_TAG = "#bullet:"

// Be careful changing these.  The code relies on checkboxes having a space either side!
const val CHECKBOX_UNTICKED_CHAR = '☐'
const val CHECKBOX_UNTICKED = " $CHECKBOX_UNTICKED_CHAR "
const val CHECKBOX_TICKED_CHAR = '☑'
const val CHECKBOX_TICKED = " $CHECKBOX_TICKED_CHAR "
const val BULLET = " • "

private val TAGS_REGEX = "#(tag:[^:]+:|check:[01]|bullet:)".toRegex()

/**
 * Transform from stick.py text to sticky string.
 */
fun fromStickyPy(
    stickyPyString: String,
    initialSpanStyle: SpanStyle = DEFAULT_INITIAL_SPAN_STYLE
): StickyString {
    val result = StickyString(initialSpanStyle)
    chunk(stickyPyString).map { chunk ->
        when {
            chunk.startsWith("#tag:") -> {
                FORMATS_BY_TAG[chunk]?.let { result.push(it) }
            }

            chunk == BULLET_TAG -> result.push(BULLET)
            chunk == CHECKBOX_OFF_TAG -> result.push(CHECKBOX_UNTICKED)
            chunk == CHECKBOX_ON_TAG -> result.push(CHECKBOX_TICKED)
            else -> result.push(chunk)
        }
    }
    return result
}

fun chunk(text: String): List<String> {
    val result = mutableListOf<String>()
    var prev = 0
    TAGS_REGEX.findAll(text).toList().map { match ->
        if (match.range.first > prev) // If there is intervening non-delimiter text.
            result.add(text.substring(prev, match.range.first))
        result.add(match.value)
        prev = match.range.last + 1
    }
    if (prev < text.length)
        result.add(text.substring(prev))
    return result
}
