/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.data

import org.grating.styncynotes.ui.richertext.Format.Companion.EMPTY_FLAGS
import org.grating.styncynotes.ui.richertext.Format.Companion.FORMATS
import org.grating.styncynotes.ui.richertext.StickyString

/**
 * Transform from sticky string to sticky.py format.
 */
fun toStickyPy(stickyString: StickyString): String {
    val stickyBldr = StringBuilder()
    var prev = EMPTY_FLAGS
    var pos = 0
    val text = stickyString.getText()
    stickyString.getFlags().forEach { range ->
        val flags = range.item
        try {
            if (range.start > pos)
                stickyBldr.append(text.substring(pos, range.start).translateNonTags())
            for (i in 0..FORMATS.lastIndex) {
                if (prev[i] != flags[i])
                    stickyBldr.append(FORMATS[i].tag)
            }
            stickyBldr.append(text.substring(range.start, range.end).translateNonTags())
            pos = range.end
            prev = flags
        } catch (e: ConversionException) {
            throw ConversionException(
                "Error @ ${range.start} for text ${
                    text.substring(range.start, range.end).translateNonTags()
                }", e)
        }
    }
    // If prev flags had stuff still switched on, switch it off or sticky.py won't understand the
    // formatting.
    for (i in 0..FORMATS.lastIndex) {
        if (prev[i] != EMPTY_FLAGS[i])
            stickyBldr.append(FORMATS[i].tag)
    }

    return stickyBldr.toString()
}

/**
 * Perform translations (or escapes) that don't start with '#tag:'
 */
private fun String.translateNonTags(): String {
    return this
        .replace(CHECKBOX_UNTICKED, "#check:0")
        .replace(CHECKBOX_TICKED, "#check:1")
        .replace(BULLET, "#bullet:")
}

class ConversionException(msg: String, cause: Exception? = null) : Exception(msg, cause)
