/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.screens

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import org.grating.styncynotes.R
import org.grating.styncynotes.data.Note
import org.grating.styncynotes.ui.model.StyncyNotesViewModel
import org.grating.styncynotes.ui.richertext.RicherTextPane
import org.grating.styncynotes.ui.richertext.getTextFieldColors

@Composable
fun NoteScreen(
    viewModel: StyncyNotesViewModel,
    onColorChangeRequested: (Note) -> Unit,
    onFormatChangeRequested: (Note) -> Unit,
    onDeleteRequested: (Note) -> Unit
) {
    val uiState = viewModel.uiState.collectAsState().value

    Column {
        NoteCard(viewModel,
                 note = uiState.note!!,
                 showFormat = true,
                 scrollable = true,
                 onColorChangeRequested = onColorChangeRequested,
                 onFormatChangeRequested = onFormatChangeRequested,
                 onDeleteRequested = onDeleteRequested)
    }
}

@Composable
fun NoteCard(
    viewModel: StyncyNotesViewModel,
    note: Note,
    showFormat: Boolean = false,
    scrollable: Boolean,
    onColorChangeRequested: (Note) -> Unit = {},
    onFormatChangeRequested: (Note) -> Unit = {},
    onDeleteRequested: (Note) -> Unit = {},
    onNoteSelected: (Note) -> Unit = {},
    maxHeight: Int = Int.MAX_VALUE
) {
    ElevatedCard(
        colors = CardDefaults.cardColors(
            contentColor = Note.TEXT_COLOR,
            containerColor = note.color.body,
        ),
        shape = RoundedCornerShape(10.dp),
        modifier = Modifier
            .padding(start = 4.dp, end = 4.dp, top = 4.dp, bottom = 4.dp)
    ) {
        Column(modifier = Modifier.heightIn(max = maxHeight.dp)) {
            Row(modifier = Modifier.height(intrinsicSize = IntrinsicSize.Max),
                verticalAlignment = Alignment.CenterVertically) {
                Column(modifier = Modifier.fillMaxHeight()) {
                    IconButton(onClick = { onColorChangeRequested(note) },
                               Modifier
                                   .background(note.color.header)
                                   .size(28.dp)
                                   .weight(1f)
                    ) {
                        Icon(painter = painterResource(R.drawable.sticky_color),
                             contentDescription = "Note Color")
                    }

                    if (showFormat) {
                        IconButton(onClick = { onFormatChangeRequested(note) }, Modifier
                            .background(note.color.header)
                            .size(28.dp)
                            .weight(1f)
                        )
                        {
                            Icon(painter = painterResource(R.drawable.sticky_text),
                                 contentDescription = "Format")
                        }
                    }
                }

                TextField(value = note.title,
                          onValueChange = {
                              viewModel.updateNoteTitle(it)
                          },
                          textStyle = MaterialTheme.typography.titleMedium,
                          colors = getTextFieldColors(Note.TEXT_COLOR, note.color.header),
                          modifier = Modifier
                              .background(note.color.header)
                              .fillMaxSize()
                              .weight(1f)
                              .align(Alignment.CenterVertically)
                )

                Column(modifier = Modifier.fillMaxHeight()) {
                    IconButton(
                        onClick = {
                            onDeleteRequested(note)
                        },
                        Modifier
                            .background(note.color.header)
                            .size(28.dp)
                            .weight(1f)
                    )
                    {
                        Icon(painter = painterResource(R.drawable.sticky_delete),
                             contentDescription = "Delete Note")
                    }
                }
            }

            RicherTextPane(viewModel = viewModel,
                           note = note,
                           foreground = Note.TEXT_COLOR,
                           background = note.color.body,
                           scrollable = scrollable,
                           onSelected = {
                               onNoteSelected(note)
                           },
                           onValueChange = { tfv ->
                               viewModel.updateNoteTfv(tfv)
                           })
        }
    }
}

