/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.richertext

import androidx.compose.ui.text.TextRange
import androidx.compose.ui.text.input.TextFieldValue
import org.grating.styncynotes.ui.data.BULLET
import org.grating.styncynotes.ui.data.CHECKBOX_TICKED
import org.grating.styncynotes.ui.data.CHECKBOX_TICKED_CHAR
import org.grating.styncynotes.ui.data.CHECKBOX_UNTICKED
import org.grating.styncynotes.ui.data.CHECKBOX_UNTICKED_CHAR
import org.grating.styncynotes.ui.richertext.RichTextEvent.CheckboxTap
import org.grating.styncynotes.ui.richertext.RichTextEvent.TypingEvent
import kotlin.math.max

/**
 * An enumeration of the things that can happen.
 */
sealed interface RichTextEvent {
    sealed interface TypingEvent : RichTextEvent
    data class AddText(val text: String, val pos: Int) : TypingEvent
    data class DelText(val start: Int, val end: Int) : TypingEvent
    data class SelectionChange(val before: TextRange, val after: TextRange) : RichTextEvent
    data class CheckboxTap(val state: CheckboxState, val pos: Int) : RichTextEvent
    data class CursorMove(val from: Int, val to: Int) : RichTextEvent
    data object Nothing : RichTextEvent
}

enum class CheckboxState(val text: String) {
    TICKED(CHECKBOX_TICKED),
    UNTICKED(CHECKBOX_UNTICKED);

    fun toggle(): CheckboxState {
        return if (this == TICKED) UNTICKED else TICKED
    }
}

/**
 * "Toggle" a checkbox within an annotated string by swapping it for the "other" version (i.e.
 * checked for unchecked and vice-versa).
 * @param tfv The TextFieldValue object returned by the GUI framework in response to the event.
 * @param sticky The stick string containing text and formatting info.
 * @param event the details of the checkbox tap event.
 * @return Text field value with the checkbox toggled.
 */
fun toggleCheckBox(tfv: TextFieldValue, sticky: StickyString, event: CheckboxTap): TextFieldValue {
    val txt = event.state.toggle().text
    sticky.replaceText(event.pos, txt)
    return tfv.copy(stickyString = sticky)
}

/**
 * "Toggle" a checklist, as opposed to a checkbox, confusingly.
 * Toggling the list means switching the list on/off (i.e. we get another checkbox every time we hit enter etc.)
 * vs toggling a checkbox which means ticking or unticking a single checkbox.
 */
fun toggleCheckList(tfv: TextFieldValue, sticky: StickyString): TextFieldValue {
    return toggleList(
        tfv = tfv,
        sticky = sticky,
        isOn = { it.startsWith(CHECKBOX_TICKED) || it.startsWith(CHECKBOX_UNTICKED) },
        removeLen = { it.prefixLen(BULLET, CHECKBOX_TICKED, CHECKBOX_UNTICKED) },
        token = CHECKBOX_UNTICKED)
}

/**
 * Toggle a bullet list.
 */
fun toggleBulletList(tfv: TextFieldValue, sticky: StickyString): TextFieldValue {
    return toggleList(
        tfv = tfv,
        sticky = sticky,
        isOn = { it.startsWith(BULLET) },
        removeLen = { it.prefixLen(BULLET, CHECKBOX_TICKED, CHECKBOX_UNTICKED) },
        token = BULLET)
}

/**
 * Toggle a bullet list or checklist (or some other list).
 */
private fun toggleList(
    tfv: TextFieldValue,
    sticky: StickyString,
    isOn: (String) -> Boolean,
    removeLen: (String) -> Int,
    token: String
): TextFieldValue {
    var pos = tfv.selection.start
    // Find start of line.
    while (pos > 0 && tfv.text[pos - 1] != '\n') pos--
    val startOfLine = tfv.text.substring(pos)
    val on = isOn(startOfLine)
    val len = removeLen(startOfLine)
    var offset = 0

    // Switching logic...
    if (len > 0) {
        if (!on) { // <------------------------------------------------- replace different
            sticky.deleteText(pos, pos + len); offset -= len
            sticky.insertText(token, pos); offset += token.length
        } else { // <--------------------------------------------------- switch off
            sticky.deleteText(pos, pos + len); offset -= len
        }
    } else { // <------------------------------------------------------- switch on
        sticky.insertText(token, pos); offset += token.length
    }

    return tfv.copy(stickyString = sticky,
                    selection = TextRange(tfv.selection.start + offset,
                                          tfv.selection.end + offset))
}

private fun String.prefixLen(@Suppress("SameParameterValue") vararg words: String): Int {
    for (word in words)
        if (startsWith(word)) return word.length
    return 0
}


/**
 * If text was added or removed, adjust the SpanStyles accordingly.
 */
fun accommodateTyping(
    tfv: TextFieldValue,
    sticky: StickyString,
    event: TypingEvent
): TextFieldValue =
    tfv.run {
        when (event) {
            is RichTextEvent.AddText -> {
                val text = addListPrefixIfRequired(sticky, event)
                val move = text.length - event.text.length
                if (move > 0)
                    copy(annotatedString = sticky.insertText(text, event.pos).annotatedString,
                         selection = TextRange(selection.start + move,
                                               selection.end + move))
                else
                    copy(annotatedString = sticky.insertText(text, event.pos).annotatedString)
            }

            is RichTextEvent.DelText ->
                copy(sticky.deleteText(event.start, event.end).annotatedString)
        }
    }

/**
 * If we're in the middle of a bullet or check list and we just pressed return, then add the
 * appropriate list symbol at the start of the line.
 * @param sticky the entire stick string.
 * @param event the AddText event containing the text that was added.
 */
private fun addListPrefixIfRequired(sticky: StickyString, event: RichTextEvent.AddText) =
    if (event.text.contains('\n')) {
        val prev = sticky.getPreviousLine(event.pos)
        if (prev.startsWith(CHECKBOX_TICKED) || prev.startsWith(CHECKBOX_UNTICKED))
            event.text.appendAfterFirst('\n', CHECKBOX_UNTICKED)
        else if (prev.startsWith(BULLET))
            event.text.appendAfterFirst('\n', BULLET)
        else
            event.text
    } else {
        event.text
    }

private fun String.appendAfterFirst(c: Char, text: String): String =
    indexOf(c).let { substring(0, it + 1) + text + substring(it + 1) }

/**
 * Try to determine what the user did that resulted in the differences between the before and
 * after text field values.
 *
 * @param before The TFV before the interaction.
 * @param after The TFV after the interaction.
 * @return The event describing what happened.
 */
fun whatHappened(
    stickyString: StickyString,
    before: TextFieldValue,
    after: TextFieldValue
): RichTextEvent {
    val beforeText = before.annotatedString.text
    val afterText = after.annotatedString.text

    return when {
        beforeText.length < afterText.length ->                                                     // Text added.
            extraText(before.selection.start,
                      afterText,
                      beforeText)

        beforeText.length > afterText.length -> {                                                   // Text deleted.
            // NOTE: The values that come out of selection aren't always as expected.  We can get
            // values that are outside the bounds of what we think the string should now be based on
            // previous inserts and deletes.  To avoid IOOBEs, constrain the selection indexes to be
            // within the bounds of what we think the string should  be (i.e. the stickString
            // contents).
            val start = after.selection.start.coerceIn(0, max(stickyString.textLastIndex, 0))
            val end = before.selection.start.coerceIn(0, max(stickyString.textLength, 0))
            if (start <= end) {
                RichTextEvent.DelText(start, end)
            } else
                RichTextEvent.Nothing // Try not to blow up if we get something unexpected from selection.
        }

        else -> {
            when (val tapState = after.tapState()) {
                is CheckboxTap -> tapState                                                          // Checkbox tapped.
                else -> {
                    if (before.selection == after.selection)                                        // Nothing happened!
                        RichTextEvent.Nothing
                    else if (before.selection.start != before.selection.end ||                      // Selection changed.
                        after.selection.start != after.selection.end
                    )
                        RichTextEvent.SelectionChange(before.selection, after.selection)
                    else
                        RichTextEvent.CursorMove(before.selection.start,
                                                 after.selection.start)                             // Cursor moved.
                }

            }
        }
    }
}

/**
 * Return the additional text and its location when comparing a larger string and a smaller one.
 * Assume that the larger and smaller strings are identical apart from a small number of added
 * characters.
 *
 * @param where Where in the two strings to look.
 * @param longer The longer of the two strings (i.e. the one with the added chars).
 * @param shorter The shorter of the two strings (i.e. the one without the added chars).
 */
private fun extraText(where: Int, longer: String, shorter: String): RichTextEvent.AddText =
    RichTextEvent.AddText(longer.substring(where, where + longer.length - shorter.length), where)


/**
 * See if the text field value equates to the user tapping on a checkbox.
 * If so, return the location and current state of the checkbox that was tapped.
 * Otherwise, return an indication that no checkbox was tapped.
 */
fun TextFieldValue.tapState(): RichTextEvent =
    if (selection.start == selection.end) {
        fun getRte(idx: Int): RichTextEvent? {
            return when {
                idx !in text.indices -> null
                text[idx] == CHECKBOX_TICKED_CHAR -> CheckboxTap(CheckboxState.TICKED, idx - 1)
                text[idx] == CHECKBOX_UNTICKED_CHAR -> CheckboxTap(CheckboxState.UNTICKED, idx - 1)
                else -> null
            }
        }
        getRte(selection.start) ?: getRte(selection.start - 1) ?: RichTextEvent.Nothing
    } else RichTextEvent.Nothing

