/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.richertext

import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextDecoration
import org.grating.styncynotes.data.replace
import org.grating.styncynotes.scaleBy
import org.grating.styncynotes.with
import org.grating.styncynotes.without

sealed interface Format {
    val name: String
    val tag: String
    val flagIdx: Int
    val flag: Char

    fun state(flags: String): State
    fun enable(spanStyle: SpanStyle, flags: String): Pair<SpanStyle, String>
    fun disable(spanStyle: SpanStyle, flags: String): Pair<SpanStyle, String>
    fun toggle(spanStyle: SpanStyle, flags: String): Pair<SpanStyle, String>

    enum class State { OFF, ON, UNSET }

    abstract class OverwritingFormat(
        override val name: String,
        override val tag: String,
        override val flag: Char
    ) :
        Base(name, tag, flag) {
        abstract fun styleOn(spanStyle: SpanStyle): SpanStyle
        abstract fun styleOff(spanStyle: SpanStyle): SpanStyle

        override fun enable(spanStyle: SpanStyle, flags: String) =
            Pair(styleOn(spanStyle), flags.enable())

        override fun disable(spanStyle: SpanStyle, flags: String) =
            Pair(styleOff(spanStyle), flags.disable())
    }


    class ScalingFormat(
        override val name: String,
        override val tag: String,
        override val flag: Char,
        private val scalar: Double
    ) : Base(name, tag, flag) {
        private val inverse: Double = 1 / scalar

        override fun enable(spanStyle: SpanStyle, flags: String): Pair<SpanStyle, String> =
            when (state(flags)) {
                State.OFF -> throw IllegalArgumentException("Enable is off.  Shouldn't try enable an off flag.")
                State.ON -> Pair(spanStyle, flags)
                State.UNSET ->
                    Pair(spanStyle.copy(fontSize = spanStyle.fontSize.scaleBy(scalar)),
                         flags.enable())
            }

        override fun disable(spanStyle: SpanStyle, flags: String): Pair<SpanStyle, String> =
            when (state(flags)) {
                State.OFF -> throw IllegalArgumentException("Enable is off.  Shouldn't try disable an off flag.")
                State.ON ->
                    Pair(spanStyle.copy(fontSize = spanStyle.fontSize.scaleBy(inverse)),
                         flags.disable())

                State.UNSET -> Pair(spanStyle, flags)
            }
    }

    abstract class Base(
        override val name: String,
        override val tag: String,
        override val flag: Char
    ) : Format {
        override val flagIdx = _FORMATS.size

        // NOTE: We leak 'this' in the initialiser before the object has completed construction.
        // This is done in order to add the newly created objects into a '_FORMATS' list so they
        // can be accessed as such later on.  By the time the _FORMATS list is accessed, it is
        // _believed_ the formats will be fully built.
        // The only alternative I can think of is to manually update a list of known formats
        // every time a new format is added, which is unnecessary manual maintenance and so to be
        // avoided.
        init {
            @Suppress("LeakingThis")
            _FORMATS.add(this)
        }

        override fun state(flags: String): State =
            when (flags[flagIdx]) {
                UNSET -> State.UNSET
                OFF -> State.OFF
                else -> State.ON
            }

        override fun toggle(spanStyle: SpanStyle, flags: String): Pair<SpanStyle, String> =
            when (state(flags)) {
                State.UNSET -> enable(spanStyle, flags)
                State.ON -> disable(spanStyle, flags)
                State.OFF -> throw IllegalArgumentException("OFF is for resetting a flag.  Shouldn't try to toggle it.")
            }

        internal fun String.enable() = replace(flagIdx, flag)
        internal fun String.disable() = replace(flagIdx, UNSET)
    }


    companion object {
        const val UNSET = '_'
        const val OFF = ' '
        const val EMPTY_FLAGS = "__________"


        private val _FORMATS = mutableListOf<Format>()

        // *** CONCRETE FORMATS ***
        val BOLD = object : OverwritingFormat("Bold", "#tag:bold:", 'B') {
            override fun styleOn(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(fontWeight = FontWeight.Bold)

            override fun styleOff(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(fontWeight = FontWeight.Normal)
        }

        @Suppress("unused")
        val ITALIC = object : OverwritingFormat("Italic", "#tag:italic:", 'I') {
            override fun styleOn(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(fontStyle = FontStyle.Italic)

            override fun styleOff(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(fontStyle = FontStyle.Normal)
        }

        val MONOSPACE = object : OverwritingFormat("Monospace", "#tag:monospace:", 'M') {
            override fun styleOn(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(fontFamily = FontFamily.Monospace)

            override fun styleOff(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(fontFamily = FontFamily.Default)
        }

        @Suppress("unused")
        val STRIKETHROUGH =
            object : OverwritingFormat("Strikethrough", "#tag:strikethrough:", 'S') {
                override fun styleOn(spanStyle: SpanStyle): SpanStyle =
                    spanStyle.copy(textDecoration = spanStyle.textDecoration?.with(TextDecoration.LineThrough)
                        ?: TextDecoration.LineThrough)

                override fun styleOff(spanStyle: SpanStyle): SpanStyle =
                    spanStyle.copy(textDecoration = spanStyle.textDecoration?.without(TextDecoration.LineThrough))
            }

        @Suppress("unused")
        val UNDERLINE = object : OverwritingFormat("Underline", "#tag:underline:", 'U') {
            override fun styleOn(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(textDecoration = spanStyle.textDecoration?.with(TextDecoration.Underline)
                    ?: TextDecoration.Underline)

            override fun styleOff(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(textDecoration = spanStyle.textDecoration?.without(TextDecoration.Underline))
        }

        val HIGHLIGHT = object : OverwritingFormat("Highlight", "#tag:highlight:", 'H') {
            override fun styleOn(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(background = Color.Yellow)

            override fun styleOff(spanStyle: SpanStyle): SpanStyle =
                spanStyle.copy(background = Color.Transparent)
        }

        val SMALL = ScalingFormat("Small text", "#tag:small:", 'S', 0.85)
        @Suppress("unused")
        val LARGE = ScalingFormat("Large text", "#tag:large:", 'L', 1.3)
        @Suppress("unused")
        val LARGER = ScalingFormat("Larger text", "#tag:larger:", 'L', 1.5)
        @Suppress("unused")
        val HEADER = ScalingFormat("Header", "#tag:header:", 'H', 1.6)

        val FORMATS: List<Format>
            get() = _FORMATS

        val FORMATS_BY_TAG = FORMATS.associateBy({ it.tag }, { it })
    }
}
