/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.model

import android.net.Uri
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.text.input.TextFieldValue
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.ViewModelProvider.AndroidViewModelFactory.Companion.APPLICATION_KEY
import androidx.lifecycle.viewModelScope
import androidx.lifecycle.viewmodel.initializer
import androidx.lifecycle.viewmodel.viewModelFactory
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import org.grating.styncynotes.StyncyNotesApplication
import org.grating.styncynotes.data.BackingStore
import org.grating.styncynotes.data.Color
import org.grating.styncynotes.data.Note
import org.grating.styncynotes.data.NotesRepository
import org.grating.styncynotes.logError2
import org.grating.styncynotes.logInfo
import org.grating.styncynotes.logInfo2
import org.grating.styncynotes.ui.data.SettingsRepository
import org.grating.styncynotes.ui.data.isUnset
import org.grating.styncynotes.ui.richertext.Format
import org.grating.styncynotes.ui.richertext.toggleBulletList
import org.grating.styncynotes.ui.richertext.toggleCheckList

private const val PENDING_DATA_STRING = ""

private const val STALE_POLL_MS = 1_000L

class StyncyNotesViewModel(
    val settingsRepository: SettingsRepository,
    private val notesRepository: NotesRepository
) : ViewModel() {
    private val _uiState = MutableStateFlow(StyncyNotesUiState())
    val uiState: StateFlow<StyncyNotesUiState> = _uiState.asStateFlow()

    private val stringSink = sinkBuilder(viewModelScope, PENDING_DATA_STRING)
    private val currentNotesFileUri = stringSink(settingsRepository.currentNotesFileUri)

    init {
        // Listen to when the selected group changes.
        viewModelScope.launch(Dispatchers.IO) {
            settingsRepository.currentGroup.collect { group ->
                logInfo("StyncyNotesViewModel", "Group changed to $group")
                if (!group.isUnset()) {
                    _uiState.update { state ->
                        state.copy(group = group, pendingGroupName = group)
                    }
                    refreshNotesCache()
                }
            }
        }

        // Routinely check if the repository has gone stale and if so, refresh our cached data.
        viewModelScope.launch(Dispatchers.IO) {
            while (true) {
                delay(STALE_POLL_MS)
                if (notesRepository.isStale()) {
                    logInfo("StyncyNotesViewModel", "Repo staleness detected.  Refreshing...")
                    refreshNotesCache()
                }
            }
        }
    }


    /**
     * Called when user selects a document URI to store the notes in.
     */
    fun setNotesFileUri(notesFileUri: Uri) {
        if (notesFileUri != currentNotesFileUri) {
            viewModelScope.launch(Dispatchers.IO) {
                settingsRepository.setNotesFileUri(notesFileUri.toString())
            }
        }
    }

    /**
     * Called by MainActivity in response to a change to the notes file URI.  This will cause the
     * notes to be read from the backing store (i.e. IO ops).
     */
    fun setBackingStore(
        backingStore: BackingStore,
        onFailure: suspend (t: Throwable) -> Unit = {}
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            try {
                notesRepository.setBackingStore(backingStore)
                val groups = notesRepository.getGroups()
                if (!groups.contains(_uiState.value.group))
                    setCurrentGroup(groups.firstOrNull() ?: "")
            } catch (e: Exception) {
                logError2("Couldn't access the underlying backing store.  You'll need to reselect",
                          e)
                onFailure(e)
            }
            refreshNotesCache()
        }
    }

    /**
     * If the given group exists, load the notes for that group into the view model state.
     */
    private suspend fun refreshNotesCache() {
        logInfo2("refreshNotesCache()::START")
        val groups = notesRepository.getAll()
        val group = _uiState.value.group
        val note = _uiState.value.note
        _uiState.update { state ->
            if (groups.containsKey(group)) {
                state.copy(groups = groups,
                           group = group,
                           notes = notesRepository.getNotesForGroup(group),
                           note = if (note != null)
                               notesRepository.getNoteForGroupAt(group, note.idx)
                           else
                               state.note
                )
            } else {
                state.copy(groups = groups)
            }
        }
        logInfo2("refreshNotesCache()::END")
    }

    /**
     * Set the group currently being viewed.
     */
    fun setCurrentGroup(group: String) {
        _uiState.update { state -> state.copy(group = group, pendingGroupName = group) }
        viewModelScope.launch(Dispatchers.IO) {
            settingsRepository.setCurrentGroup(group)
        }
    }

    /**
     * Set the note currently being viewed / edited.
     */
    fun setCurrentNote(note: Note) {
        _uiState.update { state ->
            state.copy(note = note)
        }
    }

    /**
     * Stop editing the current note and go back to viewing all notes for the current group.
     */
    private fun unsetCurrentNote() {
        _uiState.update { state ->
            state.copy(note = null)
        }
    }

    /**
     * Replace a note's text field value with the specified one.  NOTE: A text field value change
     * can occur due to something as insignificant as a cursor position change, so we should only
     * update the actual note (via the repo) if a persistable change has occurred (i.e. the note's
     * sticky string is dirty).
     * @param tfv the new TextFieldValue the note should have.
     */
    fun updateNoteTfv(tfv: TextFieldValue) {
        logInfo2("updateNoteTfv(...)::START")
        _uiState.update { state ->
            val note = state.note!!.copy(textFieldValue = tfv)
            if (note.stickyString.isDirty())
                viewModelScope.launch { notesRepository.updateNote(note) }
            state.copy(note = note)
        }
        logInfo2("updateNoteTfv(...)::END")
    }

    /**
     * Toggle checklist on and off at the start of line for the current cursor position.
     */
    fun toggleChecklist() {
        updateNoteTfv(toggleCheckList(_uiState.value.note!!.textFieldValue,
                                      _uiState.value.note!!.stickyString))
    }

    /**
     * Toggle bullets on and off at the start of line for the current cursor position.
     */
    fun toggleBullets() {
        updateNoteTfv(toggleBulletList(_uiState.value.note!!.textFieldValue,
                                       _uiState.value.note!!.stickyString))
    }

    /**
     * Change the colour of the currently selected note.
     */
    fun updateNoteColor(color: Color) {
        _uiState.update { state ->
            val note = state.note!!.copy(color = color)
            viewModelScope.launch {
                notesRepository.updateNote(note)
                refreshNotesCache()
            }
            state.copy(note = note)
        }
    }

    /**
     * Insert the specified format to be in effect over the currently selected range.
     * @param format The text format to take effect.
     */
    fun updateNoteFormat(format: Format) {
        _uiState.value.note!!.apply {
            with(textFieldValue.selection) {
                stickyString.applyFormat(format, start, end); logInfo2("$start -> $end")
            }
            updateNoteTfv(textFieldValue.copy(stickyString.annotatedString))
            logInfo2("")
        }
    }

    fun updatePendingGroupName(pendingGroupName: String) {
        _uiState.update { state ->
            state.copy(pendingGroupName = pendingGroupName)
        }
    }

    fun addNewNote() {
        viewModelScope.launch(Dispatchers.IO) {
            val newNote = notesRepository.addNewNoteToGroup(_uiState.value.group)
            refreshNotesCache()
            setCurrentNote(newNote)
        }
    }

    fun deleteNote(note: Note) {
        viewModelScope.launch(Dispatchers.IO) {
            notesRepository.deleteNote(note)
            if (_uiState.value.note == note)
                unsetCurrentNote()
            refreshNotesCache()
        }
    }

    fun updateNoteTitle(title: String) {
        logInfo2("updateNoteTitle(...)::START")
        _uiState.update { state ->
            val note = state.note!!.copy(title = title)
            viewModelScope.launch { notesRepository.updateNote(note) }
            state.copy(note = note)
        }
        logInfo2("updateNoteTitle(...)::END")
    }

    fun registerConfirmableAction(msg: String, action: () -> Unit) {
        _uiState.update { state ->
            state.copy(confirmationMsg = msg, confirmationAction = action)
        }
    }

    fun clearConfirmableAction() {
        _uiState.update { state ->
            state.copy(confirmationMsg = "", confirmationAction = {})
        }
    }

    fun addNewGroup(group: String) {
        viewModelScope.launch(Dispatchers.IO) {
            notesRepository.addNewGroup(group)
            setCurrentGroup(group)
        }
    }

    fun delGroup(group: String) {
        val groups = _uiState.value.groups.keys.toList()
        val grpIdx = groups.indexOf(group)
        if (grpIdx < 0)
            throw IllegalArgumentException("No such group '$group'")
        val newSelectedGroup = when {
            groups.size == 1 -> Note.UNSET_GROUP
            grpIdx == 0 -> groups[1]
            else -> groups[grpIdx - 1]
        }
        viewModelScope.launch(Dispatchers.IO) {
            notesRepository.deleteGroup(group)
        }
        setCurrentGroup(newSelectedGroup)
    }


    fun renameGroup(group: String, pendingGroup: String) {
        viewModelScope.launch(Dispatchers.IO) {
            notesRepository.renameGroup(group, pendingGroup)
            setCurrentGroup(pendingGroup)
        }
    }

    fun setErrorNotificationMessage(s: String) {
        _uiState.update { state ->
            state.copy(errorNotificationMessage = s)
        }
    }

    fun isNoteSelected(): Boolean = _uiState.value.note != null

    val rtFocusRequester = FocusRequester()

    companion object {
        val Factory: ViewModelProvider.Factory = viewModelFactory {
            initializer {
                val application =
                    this[APPLICATION_KEY] as StyncyNotesApplication
                StyncyNotesViewModel(application.container.settingsRepository,
                                     application.container.notesRepository)
            }
        }
    }
}

/**
 * Return a function that takes a flow and returns a state flow
 */
private fun <T> sinkBuilder(
    scope: CoroutineScope,
    @Suppress("SameParameterValue") initVal: T
): (Flow<T>) -> StateFlow<T> {
    return { flow ->
        flow.stateIn(scope = scope,
                     started = SharingStarted.WhileSubscribed(5000),
                     initialValue = initVal)
    }
}

