/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui.data

import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.stringPreferencesKey
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import org.grating.styncynotes.logInfo2


class SettingsRepository(private val dataStore: DataStore<Preferences>) {

    val currentNotesFileUri: Flow<String> =
        dataStore.data.map { preferences ->
            preferences[NOTES_FILE_URI] ?: UNSET_STRING
        }

    val currentGroup: Flow<String> =
        dataStore.data.map { preferences ->
            preferences[CURRENT_GROUP] ?: UNSET_STRING
        }

    suspend fun setNotesFileUri(uri: String) {
        dataStore.edit { preferences ->
            preferences[NOTES_FILE_URI] = uri
        }
    }

    suspend fun unsetNotesFileUri() {
        dataStore.edit { preferences ->
            preferences.remove(NOTES_FILE_URI)
        }
    }

    suspend fun setCurrentGroup(group: String) {
        logInfo2("setCurrentGroup($group)::START")
        dataStore.edit { preferences ->
            preferences[CURRENT_GROUP] = group
        }
        logInfo2("setCurrentGroup($group)::END")
    }

    private companion object {
        private val NOTES_FILE_URI = stringPreferencesKey("notes_file_uri")
        private val CURRENT_GROUP = stringPreferencesKey("current_group")
    }

}

private const val UNSET_STRING = "<UNSET>"

fun String.looksLikeDocumentUri(): Boolean {
    return this.startsWith("content://com.android.")
}

fun String.isUnset(): Boolean {
    return this == UNSET_STRING
}

