/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.ui

import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Add
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuAnchorType
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.MaterialTheme.colorScheme
import androidx.compose.material3.OutlinedIconButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusProperties
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.dialog
import androidx.navigation.compose.rememberNavController
import org.grating.styncynotes.R
import org.grating.styncynotes.highlight
import org.grating.styncynotes.ui.data.isUnset
import org.grating.styncynotes.ui.model.StyncyNotesViewModel
import org.grating.styncynotes.ui.screens.ColourChooser
import org.grating.styncynotes.ui.screens.ConfirmationDialog
import org.grating.styncynotes.ui.screens.FormatChooser
import org.grating.styncynotes.ui.screens.NoteScreen
import org.grating.styncynotes.ui.screens.NotesScreen


@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun StyncyNotesUi(
    viewModel: StyncyNotesViewModel = viewModel(factory = StyncyNotesViewModel.Factory),
    navController: NavHostController = rememberNavController()
) {
    val backStackEntry by navController.currentBackStackEntryAsState()
    // val scrollBehavior = TopAppBarDefaults.enterAlwaysScrollBehavior()
    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())
    val uiState = viewModel.uiState.collectAsState().value

    val notesFileUri: String =
        viewModel.settingsRepository.currentNotesFileUri.collectAsState("").value
    if (notesFileUri.isUnset()) NoFileUriDialog(viewModel)
    else {
        val groupsEditable = backStackEntry?.destination?.route == StyncyNotesScreen.Notes.name
        Scaffold(
            modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
            topBar = {
                StyncyNotesTopAppBar(
                    viewModel = viewModel,
                    canNavigateBack = navController.previousBackStackEntry != null,
                    groupsEditable = groupsEditable,
                    navigateUp = { navController.navigateUp() },
                    addGroup = {
                        viewModel.addNewGroup(uiState.pendingGroupName)
                    },
                    delGroup = {
                        viewModel.delGroup(uiState.group)
                    },
                    renameGroup = {
                        viewModel.renameGroup(uiState.group, uiState.pendingGroupName)
                    },
                    scrollBehavior = scrollBehavior
                )
            },
            floatingActionButton = {
                if (groupsEditable && uiState.group.isNotBlank()) {
                    FloatingActionButton(onClick = {
                        viewModel.addNewNote()
                        navController.navigate(StyncyNotesScreen.Note.name)
                    }) {
                        Icon(Icons.Filled.Add, "New Note")
                    }
                }
            }
        ) { contentPadding ->
            NavHost(
                navController = navController,
                startDestination = StyncyNotesScreen.Notes.name,
                modifier = Modifier.padding(contentPadding)
            ) {
                composable(route = StyncyNotesScreen.Notes.name) {
                    NotesScreen(viewModel = viewModel,
                                onNoteSelected = { note ->
                                    viewModel.setCurrentNote(note)
                                    navController.navigate(StyncyNotesScreen.Note.name)
                                },
                                onColorChangeRequested = { note ->
                                    viewModel.setCurrentNote(note)
                                    navController.navigate(StyncyNotesScreen.Colors.name)
                                },
                                onDeleteRequested = { note ->
                                    viewModel.registerConfirmableAction("Are you sure you want to delete this note?") {
                                        viewModel.deleteNote(note)
                                    }
                                    navController.navigate(StyncyNotesScreen.Confirm.name)
                                })
                }

                composable(route = StyncyNotesScreen.Note.name) {
                    if (viewModel.isNoteSelected()) {
                        NoteScreen(viewModel = viewModel,
                                   onColorChangeRequested = { _ ->
                                       navController.navigate(StyncyNotesScreen.Colors.name)
                                   },
                                   onFormatChangeRequested = { _ ->
                                       navController.navigate(StyncyNotesScreen.Formats.name)

                                   },
                                   onDeleteRequested = { note ->
                                       viewModel.registerConfirmableAction("Are you sure you want to delete this note?") {
                                           viewModel.deleteNote(note)
                                       }
                                       navController.navigate(StyncyNotesScreen.Confirm.name)
                                   })
                    } else {
                        navController.navigateUp()
                    }
                }

                dialog(route = StyncyNotesScreen.Colors.name) {
                    ColourChooser(onColorSelected = { color ->
                        viewModel.updateNoteColor(color)
                        navController.navigateUp()
                    })
                }

                dialog(route = StyncyNotesScreen.Formats.name) {
                    // @formatter:off - IntelliJ formatter is not working in a reasonable way.
                    val swkb = LocalSoftwareKeyboardController.current
                    FormatChooser(onFormatSelected = {
                                      viewModel.updateNoteFormat(it)
                                      navController.navigateUp()
                                      viewModel.rtFocusRequester.requestFocus()
                                      if (uiState.softKeyboardInUse) swkb?.show()
                                   },
                                  onToggleCheckboxList = {
                                      viewModel.toggleChecklist()
                                      navController.navigateUp()
                                      viewModel.rtFocusRequester.requestFocus()
                                  },
                                  onToggleBulletList = {
                                      viewModel.toggleBullets()
                                      navController.navigateUp()
                                      viewModel.rtFocusRequester.requestFocus()
                                      if (uiState.softKeyboardInUse) swkb?.show()
                                  })
                    // @formatter:on - IntelliJ formatter is not working in a reasonable way.
                }

                dialog(route = StyncyNotesScreen.Confirm.name) {
                    ConfirmationDialog(msg = uiState.confirmationMsg,
                                       onOk = {
                                           uiState.confirmationAction()
                                           navController.navigateUp()
                                       },
                                       onCancel = {
                                           viewModel.clearConfirmableAction()
                                           navController.navigateUp()
                                       })
                }
            }
        }
    }
}

enum class StyncyNotesScreen {
    Note,
    Notes,
    Colors,
    Formats,
    Confirm
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun StyncyNotesTopAppBar(
    viewModel: StyncyNotesViewModel,
    canNavigateBack: Boolean,
    groupsEditable: Boolean,
    navigateUp: () -> Unit,
    addGroup: () -> Unit,
    delGroup: () -> Unit,
    renameGroup: () -> Unit,
    modifier: Modifier = Modifier,
    scrollBehavior: TopAppBarScrollBehavior
) = CenterAlignedTopAppBar(
    scrollBehavior = scrollBehavior,
    title = {
        GroupTextField(viewModel = viewModel,
                       groupsEditable = groupsEditable,
                       addGroup,
                       delGroup,
                       renameGroup)
    },
    colors = TopAppBarDefaults.mediumTopAppBarColors(
        containerColor = colorScheme.primaryContainer
    ),
    modifier = modifier.fillMaxWidth(),
    navigationIcon = {
        if (canNavigateBack) {
            IconButton(onClick = navigateUp) {
                Icon(
                    imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                    contentDescription = stringResource(R.string.back_button)
                )
            }
        } else {
            Spacer(modifier = Modifier.padding(24.dp))
        }
    },
    actions = {
        Spacer(modifier = Modifier.padding(24.dp))
    }
)

@Composable
@OptIn(ExperimentalMaterial3Api::class)
private fun GroupTextField(
    viewModel: StyncyNotesViewModel,
    groupsEditable: Boolean,
    onAddGroup: () -> Unit,
    onDeleteGroup: () -> Unit,
    onRenameGroup: () -> Unit
) {
    val uiState = viewModel.uiState.collectAsState().value
    var dropdownExpanded by remember { mutableStateOf(false) }
    val keyboardController = LocalSoftwareKeyboardController.current
    val focusRequester = remember { FocusRequester() }
    val focusManager = LocalFocusManager.current

    if (!groupsEditable) dropdownExpanded = false
    ExposedDropdownMenuBox(expanded = dropdownExpanded,
                           onExpandedChange = {
                               dropdownExpanded = it
                           },
                           modifier = Modifier.fillMaxWidth()) {
        OutlinedTextField(
            value = uiState.pendingGroupName,
            onValueChange = {
                viewModel.updatePendingGroupName(it)
            },
            label = {
                if (uiState.pendingGroupName.isBlank())
                    Text(text = "Group")
            },
            leadingIcon = {
                if (groupsEditable) {
                    OutlinedIconButton(
                        onClick = onRenameGroup,
                        enabled = uiState.group != uiState.pendingGroupName
                    ) {
                        Icon(painter = painterResource(R.drawable.sticky_edit),
                             contentDescription = "Rename group",
                             modifier = Modifier.padding(2.dp))
                    }
                }
            },
            trailingIcon = {
                if (groupsEditable) {
                    if (uiState.group != uiState.pendingGroupName) {
                        OutlinedIconButton(onClick = onAddGroup) {
                            Icon(painter = painterResource(R.drawable.sticky_add),
                                 contentDescription = "Add new group",
                                 modifier = Modifier.padding(2.dp))
                        }
                    } else {
                        OutlinedIconButton(
                            onClick = onDeleteGroup,
                            enabled = uiState.groups.containsKey(uiState.group) && uiState.notes.isEmpty()
                        ) {
                            Icon(painter = painterResource(R.drawable.sticky_delete),
                                 contentDescription = "Delete group",
                                 modifier = Modifier.padding(2.dp))
                        }
                    }
                }
            },
            colors = if (uiState.group != uiState.pendingGroupName) {
                OutlinedTextFieldDefaults.colors(
                    focusedContainerColor = colorScheme.surfaceBright.highlight(Color.Red),
                    unfocusedContainerColor = colorScheme.surface.highlight(Color.Red),
                    disabledContainerColor = colorScheme.surfaceDim.highlight(Color.Red),
                )
            } else {
                OutlinedTextFieldDefaults.colors(
                    focusedContainerColor = colorScheme.surfaceBright,
                    unfocusedContainerColor = colorScheme.surface,
                    disabledContainerColor = colorScheme.surfaceDim,
                )
            },
            textStyle = MaterialTheme.typography.titleLarge.copy(textAlign = TextAlign.Center),
            modifier = Modifier
                .padding(start = 16.dp, end = 16.dp, top = 12.dp, bottom = 12.dp)
                .menuAnchor(ExposedDropdownMenuAnchorType.PrimaryEditable, true)
                .focusRequester(focusRequester)
                .fillMaxWidth(),
            enabled = groupsEditable,
            readOnly = !groupsEditable,
            singleLine = false,
            keyboardOptions = KeyboardOptions(imeAction = ImeAction.Search),
            keyboardActions = KeyboardActions(onDone = {
                dropdownExpanded = false
                keyboardController?.hide()
            }),
        )

        ExposedDropdownMenu(expanded = dropdownExpanded,
                            onDismissRequest = {
                                dropdownExpanded = false
                            },
                            modifier = Modifier
                                .padding(start = 16.dp,
                                         end = 16.dp,
                                         top = 12.dp,
                                         bottom = 12.dp)
                                .focusProperties { canFocus = false }
        ) {
            uiState.groups.keys.forEach {
                DropdownMenuItem(
                    text = { Text(text = it, style = MaterialTheme.typography.headlineSmall) },
                    onClick = {
                        viewModel.setCurrentGroup(it)
                        focusManager.clearFocus()
                        dropdownExpanded = false
                    }
                )
            }
        }
    }
}
