/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.data

import android.content.ContentResolver
import androidx.compose.ui.text.input.TextFieldValue
import androidx.documentfile.provider.DocumentFile
import androidx.lifecycle.LifecycleCoroutineScope
import androidx.lifecycle.ProcessLifecycleOwner
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.grating.styncynotes.add
import org.grating.styncynotes.at
import org.grating.styncynotes.copy
import org.grating.styncynotes.del
import org.grating.styncynotes.logInfo2
import org.grating.styncynotes.ui.data.fromStickyPy
import org.grating.styncynotes.ui.data.toStickyPy
import java.lang.System.currentTimeMillis
import java.util.concurrent.atomic.AtomicInteger
import kotlin.random.Random

private const val INIT = -1L

/**
 * Main repo for providing (and persisting change to) notes data.
 */
class NotesRepository(
    private var backingStore: BackingStore = BackingStore.STUB,
    lifecycleScope: LifecycleCoroutineScope = ProcessLifecycleOwner.get().lifecycleScope
) {
    private val mutex = Mutex()
    private val saveRequest = MutableStateFlow(INIT)
    private var lastRefresh = 0L
    private lateinit var cache: Map<String, List<Note>>
    private val pendingSaveRequests = AtomicInteger(0)

    init {
        // ** Saving Loop **
        // Wait for SAVE_DELAY_MS of no changes before saving, so as not to wear out the sd card.
        lifecycleScope.launch {
            saveRequest.collect { saveRequestMs ->
                if (saveRequestMs == INIT) return@collect // Don't bother if startup noise.
                pendingSaveRequests.incrementAndGet()
                delay(saveRequestMs + SAVE_DELAY_MS - currentTimeMillis())
                if (saveRequestMs == saveRequest.value)
                    saveAll()
                pendingSaveRequests.decrementAndGet()
            }
        }
    }

    /**
     * Determine if the copy we have in memory is older than the touch time on the underlying backing-
     * store (i.e. the data changed underneath a running instance).
     */
    suspend fun isStale(): Boolean {
        return lastRefresh < backingStore.lastModified()
    }

    /**
     * Set a new backing store for persisting the notes data.
     */
    suspend fun setBackingStore(backingStore: BackingStore) {
        logInfo2("setBackingStore($backingStore)::START")
        awaitPendingSaves()
        if (backingStore != this.backingStore) {
            this.backingStore = backingStore
            refresh()
        }
        logInfo2("setBackingStore(...)::END")
    }

    /**
     * Don't return until all pending saves have been processed.
     */
    private suspend fun awaitPendingSaves() {
        while (pendingSaveRequests.get() > 0) {
            logInfo2("Waiting for pending saves...")
            delay(1000)
        }
    }

    /**
     * Get all groups of notes, refreshing from backing store if necessary.
     */
    suspend fun getAll(): Map<String, List<Note>> {
        logInfo2("getNotes()::START")
        if (isStale())
            refresh()
        logInfo2("getNotes()::END")
        return cache
    }

    suspend fun getGroups(): Set<String> = getAll().keys

    /**
     * Add a new note with default color and size.
     * Set random positioning for display on desktop if also using Sticky.py notes.
     * Ensure transient fields are set correctly for new notes position in group.
     *
     * @param group the group to which the new note should be added.
     */
    suspend fun addNewNoteToGroup(group: String): Note {
        logInfo2("addNewNote()::START")
        val note = Note(x = Random.nextInt(0, 1000),
                        y = Random.nextInt(0, 1000),
                        height = 200,
                        width = 200,
                        color = Color.YELLOW,
                        title = "New Note",
                        text = "")
        cache = getAll().let { groups ->
            groups.copy(group to groups[group].let { group ->
                group!!.add(note)
            })
        }
        note.setTransientFields(group, cache[group]!!.lastIndex)
        logInfo2("addNewNote()::END")
        eventuallySave()
        return note
    }

    /**
     * Delete the supplied note from its group.
     */
    suspend fun deleteNote(note: Note) {
        logInfo2("delNote(${note.title})::START")
        cache = getAll().let { groups ->
            groups.copy(note.group to groups[note.group].let { group ->
                group!!.del(note.idx).mapIndexed { index, note ->
                    note.idx = index // Have to adjust the notes indexes because of delete.
                    note
                }
            })
        }
        logInfo2("delNote(${note.title})::END")
        eventuallySave()
    }

    /**
     * Save changes to a note.
     */
    suspend fun updateNote(note: Note) {
        logInfo2("updateNote(${note.title})::START")
        cache = getAll().let { groups ->
            groups.copy(note.group to groups[note.group].let { group ->
                group!!.copy(note at note.idx)
            })
        }
        logInfo2("updateNote(${note.title})::END")
        eventuallySave()
    }

    suspend fun groupExists(group: String): Boolean = getAll().containsKey(group)

    suspend fun getNotesForGroup(group: String): List<Note> =
        getAll()[group] ?: emptyList()

    suspend fun getNoteForGroupAt(group: String, idx: Int): Note? =
        getNotesForGroup(group).getOrNull(idx)

    suspend fun addNewGroup(group: String) {
        logInfo2("addNewGroup($group)::START")
        if (groupExists(group))
            throw IllegalArgumentException("Group '$group' already exists.")
        cache = cache.copy(group to emptyList())
        eventuallySave()
        logInfo2("addNewGroup($group)::END")
    }

    suspend fun deleteGroup(group: String) {
        logInfo2("deleteGroup($group)::START")

        if (!groupExists(group))
            throw IllegalArgumentException("Group '$group' does not exist.")
        if (getNotesForGroup(group).isNotEmpty())
            throw IllegalArgumentException("Group '$group' is not empty.")

        cache = cache.del(group)
        eventuallySave()
        logInfo2("deleteGroup($group)::END")
    }

    suspend fun renameGroup(group: String, pendingGroup: String) {
        logInfo2("renameGroup($group, $pendingGroup)::START")
        if (!groupExists(group))
            throw IllegalArgumentException("Group '$group' does not exist.")
        cache = cache.map {(if (it.key == group) pendingGroup else it.key) to it.value }.toMap()
        eventuallySave()
        logInfo2("renameGroup($group, $pendingGroup)::END")
    }

    /**
     * Save all groups/notes back to backing store.
     */
    private suspend fun saveAll() {
        logInfo2("saveAll()::START")

        // Convert the annotated string rep back into stick.py before saving.
        cache.forEach { group ->
            group.value.forEach { note ->
                if (note.stickyString.isDirty()) {
                    note.text = toStickyPy(note.stickyString)
                    note.stickyString.resetDirty()
                }
            }
        }

        // And save.
        val jsonStr = JSON.encodeToString(cache)
        backingStore.writeText(jsonStr)
        lastRefresh = backingStore.lastModified()
        logInfo2("saveAll()::END")
    }

    /**
     * Refresh our cache from the underlying backing store.
     */
    private suspend fun refresh() {
        mutex.withLock {
            awaitPendingSaves()
            logInfo2("refresh()::START")
            cache = JSON.decodeFromString<Map<String, List<Note>>>(
                backingStore.readText().ifBlank { "{}" }
            )
            logInfo2("cache.size = ${cache.size}")
            cache.entries.forEach { (group, notes) ->
                notes.forEachIndexed { idx, note -> note.setTransientFields(group, idx) }
            }
            lastRefresh = backingStore.lastModified()
            logInfo2("refresh()::END")
        }
    }

    /**
     * Set a notes transient fields.
     * @param group the group to which the note belongs.
     * @param idx the position of the note within the group.
     */
    private fun Note.setTransientFields(group: String, idx: Int) {
        this.group = group
        this.idx = idx
        this.stickyString = fromStickyPy(text)
        textFieldValue = TextFieldValue(stickyString.annotatedString)
        stickyString.resetDirty()
    }

    /**
     * Just for readability.
     */
    private fun eventuallySave() {
        saveRequest.update { currentTimeMillis() }
    }

    companion object {
        val JSON = Json {
            prettyPrint = true
        }
        const val SAVE_DELAY_MS = 5000L
    }
}

/**
 * Classes which save notes data to permanent storage should implement this interface.
 */
interface BackingStore {
    suspend fun readText(): String
    suspend fun writeText(text: String)
    suspend fun lastModified(): Long

    companion object {
        private val LAST_MODIFIED = currentTimeMillis()

        val STUB: BackingStore = object : BackingStore {
            override suspend fun readText(): String = "{}"
            override suspend fun writeText(text: String) {}
            override suspend fun lastModified() = LAST_MODIFIED
        }
    }
}

/**
 * Backing store that uses a document file to store notes data.
 */
data class ExternalFile(
    private val contentResolver: ContentResolver,
    private val docFile: DocumentFile
) : BackingStore {
    override suspend fun readText(): String {
        return contentResolver.openInputStream(docFile.uri)?.bufferedReader().use {
            it?.readText()
        }!!
    }

    override suspend fun writeText(text: String) {
        contentResolver.openOutputStream(docFile.uri, "wt")?.bufferedWriter().use {
            it?.write(text)
        }
    }

    override suspend fun lastModified(): Long {
        return docFile.lastModified()
    }
}

class FakeBackingStore : BackingStore {
    override suspend fun readText(): String {
        return JSON_AS_STRING
    }

    override suspend fun writeText(text: String) {
        throw Exception("Writing to fake backing store is unsupported.")
    }

    override suspend fun lastModified(): Long {
        return 0
    }

    companion object {
        private val JSON_AS_STRING = """
            {
                "General": [
                    {
                        "x": 1272,
                        "y": 367,
                        "height": 337,
                        "width": 505,
                        "color": "blue",
                        "title": "Household ToDo",
                        "text": ""
                    },
                    {
                        "x": 92,
                        "y": 497,
                        "height": 338,
                        "width": 432,
                        "color": "orange",
                        "title": "Immediate Attention",
                        "text": ""
                    },
                    {
                        "x": 634,
                        "y": 569,
                        "height": 360,
                        "width": 423,
                        "color": "red",
                        "title": "Today",
                        "text": ""
                    }
                ],
                "Development": [
                    {
                        "x": 1184,
                        "y": 170,
                        "height": 492,
                        "width": 524,
                        "color": "blue",
                        "title": "RecollDroid Development",
                        "text": ""
                    }
                ]
            }
        """.trimIndent()
    }

}


suspend fun main() {
    val repo = NotesRepository(FakeBackingStore())
    val groups = repo.getAll()
    print(groups)
}