/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes.data

import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.input.TextFieldValue
import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.Transient
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import org.grating.styncynotes.ui.richertext.StickyString
import androidx.compose.ui.graphics.Color as KotlinColor

enum class Color(val header: KotlinColor, val body: KotlinColor) {
    BLUE(KotlinColor(0x3d, 0x9b, 0xff, 0xff),
         KotlinColor(0x81, 0xbd, 0xff, 0xff)),
    GREEN(KotlinColor(0x67, 0xff, 0x67, 0xff),
          KotlinColor(0xaf, 0xff, 0xaf, 0xff)),
    MAGENTA(KotlinColor(0xff, 0x7f, 0xf7, 0xff),
            KotlinColor(0xff, 0xb9, 0xfb, 0xff)),
    ORANGE(KotlinColor(0xff, 0xa9, 0x39, 0xff),
           KotlinColor(0xff, 0xc4, 0x7b, 0xff)),
    PURPLE(KotlinColor(0xa5, 0x53, 0xff, 0xff),
           KotlinColor(0xc6, 0x95, 0xff, 0xff)),
    RED(KotlinColor(0xff, 0x55, 0x61, 0xff),
        KotlinColor(0xff, 0x89, 0x90, 0xff)),
    TEAL(KotlinColor(0x41, 0xff, 0xed, 0xff),
         KotlinColor(0xaf, 0xff, 0xf8, 0xff)),
    YELLOW(KotlinColor(0xf6, 0xf9, 0x07, 0xff),
           KotlinColor(0xfe, 0xff, 0x9f, 0xff))
}


fun String.replace(idx: Int, ch: Char) = String(toCharArray().also { it[idx] = ch })


@Serializable
data class Note(
    val x: Int,
    val y: Int,
    val height: Int,
    val width: Int,
    @Serializable(with = ColorSerializer::class)
    val color: Color,
    val title: String,
    var text: String,

    // 'Note' related scratch fields.
    @Transient
    var stickyString: StickyString = StickyString(),
    @Transient
    var textFieldValue: TextFieldValue = UNSET_TEXT_FIELD_VALUE,
    @Transient
    var group: String = UNSET_GROUP,
    @Transient
    var idx: Int = UNSET_IDX,
) {
    companion object {
        const val UNSET_GROUP = ""
        const val UNSET_IDX = Integer.MIN_VALUE
        val UNSET_TEXT_FIELD_VALUE = TextFieldValue(AnnotatedString(""))
        val TEXT_COLOR = KotlinColor(0x40, 0x40, 0x40, 0xff)
    }
}

object ColorSerializer : KSerializer<Color> {
    override val descriptor: SerialDescriptor
        get() = PrimitiveSerialDescriptor("Color", PrimitiveKind.STRING)

    override fun deserialize(decoder: Decoder): Color {
        return Color.valueOf(decoder.decodeString().uppercase())
    }

    override fun serialize(encoder: Encoder, value: Color) {
        encoder.encodeString(value.name.lowercase())
    }
}

