/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes

import android.util.Log
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.compositeOver
import kotlin.math.abs
import kotlin.math.floor

/**
 * Highlight a colour by first ensuring the alpha isn't completely opaque and then rendering over
 * some specified colour.
 * @param c Color to highlight with.
 * @return the highlighted colour.
 */
fun Color.highlight(c: Color): Color {
    return Color(this.red, this.green, this.blue, 0.8f).compositeOver(c)
}

@Suppress("unused")
fun logError(tag: Any, msg: Any) {
    Log.e(tag.toString(), msg.toString())
}

@Suppress("unused")
fun logError(tag: Any, msg: Any, e: Throwable) {
    Log.e(tag.toString(), msg.toString(), e)
}

fun logInfo(tag: Any, msg: Any) {
    Log.i(tag.toString(), msg.toString())
}

fun Any.logInfo2(msg: Any) {
    Log.i(this.javaClass.simpleName, msg.toString())
}

@Suppress("unused")
fun Any.logWarn2(msg: Any) {
    Log.w(this.javaClass.simpleName, msg.toString())
}

fun Any.logError2(msg: Any, t: Throwable) {
    Log.i(this.javaClass.simpleName, msg.toString(), t)
}

infix fun Float.equalsIsh(b: Float): Boolean = abs(this - b) < 1e-6

fun <K, V> Map<K, V>.copy(vararg pairs: Pair<K, V>): Map<K, V> =
    toMutableMap().let { map ->
        pairs.forEach { map[it.first] = it.second }
        map
    }

fun <K, V> Map<K, V>.del(vararg keys: K): Map<K, V> =
    toMutableMap().let { map ->
        keys.forEach { map.remove(it) }
        map
    }

fun <T> List<T>.copy(vararg pairs: Pair<Int, T>): List<T> =
    toMutableList().let { list ->
        pairs.forEach {
            list[it.first] = it.second
        }
        list
    }

fun <T> List<T>.add(vararg items: T): List<T> =
    toMutableList().let { list ->
        items.forEach {
            list.add(it)
        }
        list
    }

fun <T> List<T>.del(vararg idxs: Int): List<T> =
    toMutableList().let { list ->
        idxs.sortedDescending().forEach { list.removeAt(it) }
        list
    }


/**
 * For creating pairs representing an item at a position (i.e. in an order sequence).
 */
infix fun <A, B> A.at(that: B): Pair<B, A> = Pair(that, this)

fun Int.ifNeg(block: (Int) -> Int): Int = if (this < 0) block(this) else this

fun Double.toNdp(dp: Int): Double {
    var pow = 10
    for (i in 1..dp)
        pow *= 10
    val tmp = this * pow
    return floor(if (tmp - floor(tmp) >= 0.5f) tmp + 1 else tmp) / pow
}

