/* Copyright (C) 2024 Graham Bygrave
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package org.grating.styncynotes

import android.net.Uri
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.Surface
import androidx.compose.runtime.SideEffect
import androidx.compose.ui.Modifier
import androidx.core.view.ViewCompat
import androidx.core.view.WindowCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.updatePadding
import androidx.documentfile.provider.DocumentFile
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.viewmodel.compose.viewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.grating.styncynotes.data.ExternalFile
import org.grating.styncynotes.ui.StyncyNotesUi
import org.grating.styncynotes.ui.data.looksLikeDocumentUri
import org.grating.styncynotes.ui.model.StyncyNotesViewModel
import org.grating.styncynotes.ui.theme.StyncyNotesTheme

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()

        ViewCompat.setOnApplyWindowInsetsListener(findViewById(android.R.id.content)){ view, insets ->
            val bottom = insets.getInsets(WindowInsetsCompat.Type.ime()).bottom
            view.updatePadding(bottom = bottom)
            insets
        }

        setContent {
            val viewModel: StyncyNotesViewModel = viewModel(factory = StyncyNotesViewModel.Factory)
            StyncyNotesTheme {
                Surface(
                    modifier = Modifier.fillMaxSize()
                ) {
                    WindowCompat.setDecorFitsSystemWindows(window, false)
                    StyncyNotesUi(viewModel)
                    SideEffect {
                        registerCallbackForChangeOfBackingStore(viewModel)
                    }
                }
            }
        }
    }

    /**
     * Sit listening to the settings repository for changes to the currentNoteFileUri setting.  When
     * we get notified of a change, build a DocumentFile from this Uri and use this in turn to build
     * an ExternalFile based backing store.
     *
     * Why is this located here?
     * This code needs access to:
     * a) The ViewModel.
     * b) The Context.
     * So it clearly can't go in the view model (shouldn't put context in the view model).
     * It can't go in the repository code either - this has (and should have) no access to the
     * viewmodel.
     * This feels like a crap place for it, but I can't think of anywhere else that would be better.
     */
    private fun registerCallbackForChangeOfBackingStore(viewModel: StyncyNotesViewModel) {
        lifecycleScope.launch(Dispatchers.IO) {
            viewModel.settingsRepository.currentNotesFileUri.collect {
                if (it.looksLikeDocumentUri() && it != prevRepo) {
                    val docFile = DocumentFile.fromSingleUri(applicationContext, Uri.parse(it))!! // only null if SDK < 19
                    suspend fun onError(t:Throwable) {
                        viewModel.setErrorNotificationMessage(t.message ?: "Error setting file.")
                        viewModel.settingsRepository.unsetNotesFileUri()
                    }
                    viewModel.setBackingStore(ExternalFile(contentResolver, docFile), ::onError)
                    prevRepo = it
                }
            }
        }
    }
    private var prevRepo: String = ""
}
