package org.fossify.clock.services

import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.Service
import android.content.Context
import android.content.Intent
import android.os.Handler
import android.os.IBinder
import android.os.Looper
import androidx.core.app.NotificationCompat
import androidx.core.content.ContextCompat
import org.fossify.clock.R
import org.fossify.clock.extensions.getFormattedDuration
import org.fossify.clock.extensions.getOpenTimerTabIntent
import org.fossify.clock.extensions.timerHelper
import org.fossify.clock.helpers.INVALID_TIMER_ID
import org.fossify.clock.helpers.TIMER_RUNNING_NOTIFICATION_ID
import org.fossify.clock.models.TimerEvent
import org.fossify.clock.models.TimerState
import org.fossify.commons.extensions.notificationManager
import org.fossify.commons.extensions.showErrorToast
import org.greenrobot.eventbus.EventBus
import org.greenrobot.eventbus.Subscribe
import org.greenrobot.eventbus.ThreadMode

class TimerService : Service() {
    private val bus = EventBus.getDefault()
    private var isStopping = false

    override fun onCreate() {
        super.onCreate()
        bus.register(this)
    }

    override fun onBind(intent: Intent?): IBinder? = null

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        super.onStartCommand(intent, flags, startId)
        isStopping = false
        updateNotification()
        startForeground(
            TIMER_RUNNING_NOTIFICATION_ID,
            notification(
                title = getString(R.string.app_name),
                contentText = getString(R.string.timers_notification_msg),
                firstRunningTimerId = INVALID_TIMER_ID
            )
        )
        return START_NOT_STICKY
    }

    private fun updateNotification() {
        timerHelper.getTimers { timers ->
            val runningTimers = timers.filter { it.state is TimerState.Running }
            if (runningTimers.isNotEmpty()) {
                val firstTimer = runningTimers.first()
                val formattedDuration =
                    (firstTimer.state as TimerState.Running).tick.getFormattedDuration()
                val contextText = when {
                    firstTimer.label.isNotEmpty() -> getString(
                        R.string.timer_single_notification_label_msg,
                        firstTimer.label
                    )

                    else -> resources.getQuantityString(
                        R.plurals.timer_notification_msg,
                        runningTimers.size,
                        runningTimers.size
                    )
                }

                Handler(Looper.getMainLooper()).post {
                    try {
                        startForeground(
                            TIMER_RUNNING_NOTIFICATION_ID,
                            notification(
                                title = formattedDuration,
                                contentText = contextText,
                                firstRunningTimerId = firstTimer.id!!
                            )
                        )
                    } catch (e: Exception) {
                        showErrorToast(e)
                    }
                }
            } else {
                stopService()
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onMessageEvent(event: TimerStopService) {
        stopService()
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onMessageEvent(event: TimerEvent.Refresh) {
        if (!isStopping) {
            updateNotification()
        }
    }

    private fun stopService() {
        isStopping = true
        stopForeground(STOP_FOREGROUND_REMOVE)
    }

    override fun onDestroy() {
        super.onDestroy()
        bus.unregister(this)
    }

    private fun notification(
        title: String,
        contentText: String,
        firstRunningTimerId: Int,
    ): Notification {
        val channelId = "simple_alarm_timer"
        val label = getString(R.string.timer)
        val importance = NotificationManager.IMPORTANCE_DEFAULT
        NotificationChannel(channelId, label, importance).apply {
            setSound(null, null)
            notificationManager.createNotificationChannel(this)
        }

        val builder = NotificationCompat.Builder(this, channelId)
            .setContentTitle(title)
            .setContentText(contentText)
            .setSmallIcon(R.drawable.ic_hourglass_vector)
            .setPriority(Notification.PRIORITY_DEFAULT)
            .setSound(null)
            .setOngoing(true)
            .setAutoCancel(true)
            .setChannelId(channelId)
            .setForegroundServiceBehavior(NotificationCompat.FOREGROUND_SERVICE_IMMEDIATE)

        if (firstRunningTimerId != INVALID_TIMER_ID) {
            builder.setContentIntent(this.getOpenTimerTabIntent(firstRunningTimerId))
        }

        builder.setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
        return builder.build()
    }
}

fun startTimerService(context: Context) {
    Handler(Looper.getMainLooper()).post {
        try {
            ContextCompat.startForegroundService(context, Intent(context, TimerService::class.java))
        } catch (e: Exception) {
            context.showErrorToast(e)
        }
    }
}

object TimerStopService
