package org.fdroid.ui.crash

import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Arrangement.spacedBy
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.input.TextFieldState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Button
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.semantics.hideFromAccessibility
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import org.fdroid.R
import org.fdroid.ui.FDroidContent

@Composable
fun CrashContent(
    isOldCrash: Boolean,
    onCancel: () -> Unit,
    onSend: (String, String) -> Unit,
    textFieldState: TextFieldState,
    modifier: Modifier = Modifier
) {
    val scrollState = rememberScrollState()
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.SpaceBetween,
        modifier = modifier
            .fillMaxSize()
            .imePadding()
            .padding(horizontal = 16.dp)
            .verticalScroll(scrollState)
    ) {
        Image(
            painter = painterResource(id = R.drawable.ic_crash),
            colorFilter = ColorFilter.tint(MaterialTheme.colorScheme.primary),
            contentDescription = null, // decorative element
            modifier = Modifier
                .fillMaxWidth(0.5f)
                .aspectRatio(1f)
                .padding(vertical = 16.dp)
                .semantics { hideFromAccessibility() },
        )
        Column(verticalArrangement = spacedBy(16.dp)) {
            Text(
                text = if (isOldCrash) {
                    stringResource(R.string.crash_dialog_title_old)
                } else {
                    stringResource(R.string.crash_dialog_title)
                },
                style = MaterialTheme.typography.titleLarge,
                modifier = Modifier.align(Alignment.CenterHorizontally)
            )
            Text(
                text = stringResource(R.string.crash_report_text),
                style = MaterialTheme.typography.bodyLarge,
            )
            if (!isOldCrash) TextField(
                state = textFieldState,
                placeholder = { Text(stringResource(R.string.crash_report_comment_hint)) },
                modifier = Modifier.fillMaxWidth()
            )
        }
        Row(
            horizontalArrangement = Arrangement.SpaceEvenly,
            modifier = Modifier
                .fillMaxWidth()
                .padding(vertical = 16.dp)
        ) {
            OutlinedButton(onClick = onCancel) {
                Text(stringResource(R.string.cancel))
            }
            Button(onClick = {
                val text = if (isOldCrash) "old crash" else textFieldState.text.toString()
                onSend(text, "")
            }) {
                Text(stringResource(R.string.crash_report_button_send))
            }
        }
    }
}

@Preview
@Composable
private fun Preview() {
    FDroidContent {
        Crash(false, {}, { _, _ -> }, { _, _ -> true })
    }
}

@Preview
@Composable
private fun PreviewOld() {
    FDroidContent {
        Crash(true, {}, { _, _ -> }, { _, _ -> true })
    }
}
