package org.eagsoftware.laundrynotes.activities;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.provider.OpenableColumns;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ImageButton;
import android.widget.ProgressBar;
import android.widget.Spinner;

import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.ActivityResult;
import androidx.activity.result.ActivityResultCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.app.AppCompatDelegate;
import androidx.core.os.LocaleListCompat;

import com.google.android.material.snackbar.Snackbar;

import org.eagsoftware.laundrynotes.BackupDBManager;
import org.eagsoftware.laundrynotes.MyPrefMgr;
import org.eagsoftware.laundrynotes.R;
import org.eagsoftware.laundrynotes.Utilities;
import org.eagsoftware.laundrynotes.adapters.LanguageSpinnerAdapter;
import org.eagsoftware.laundrynotes.data.LavanderiaDB;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;


public class SettingsActivity extends AppCompatActivity {
    private static final String BACKUP_EXTENSION = ".bkp";

    Spinner mSpnLng;
    View mSetView;
    private ActivityResultLauncher<Intent> mArlExp;
    private ActivityResultLauncher<Intent> mArlImp;
    ProgressBar mPrbLoad;
    OnBackPressedCallback mNoBackCallback;
    


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_settings);

        int colID = Utilities.getIntColorFromAttr(this, android.R.attr.statusBarColor);
        Utilities.setSystemBarsAppearance(getWindow(), colID);

        mSetView = findViewById(android.R.id.content);
        mSpnLng = findViewById(R.id.spn_set_lng);
        mPrbLoad = findViewById(R.id.prb_set_load);

        setSpinnerLingue();
        setActivityResult();
        setBackupListener();
        setBackPressed();
    }


    @Override
    public void finish() {
        super.finish();
        overridePendingTransition(R.anim.fade_in, R.anim.slide_out_left);
    }



    /*-----------------------------------------------------------------------------------------------------*/
    /* METODI PRIVATI */


    private void setSpinnerLingue(){
        // Trova le lingue disponibili dai file values/strings e popola la spinner

        LanguageSpinnerAdapter mSpnLngAdp = new LanguageSpinnerAdapter(this,
                Utilities.lingueDisponibiliInApp(this));
        mSpnLng.setAdapter(mSpnLngAdp);

        // Seleziona la lingua corretta nella spinner in base al valore salvato nelle impostazioni.
        mSpnLngAdp.setSelectionByCode(mSpnLng, MyPrefMgr.getPrefLng(this));

        mSpnLng.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                // Recupera il codice lingua dall'adapter
                String newLng = ((LanguageSpinnerAdapter) parent.getAdapter()).getItem(position);
                if (newLng == null) {
                    Utilities.getCopyErrSnb(mSetView, getString(R.string.errore_generico),
                            "Impossibile ottenere la lingua selezionata dalla spinner - newLng is null");
                    return;
                }
                // Se è uguale a quello già salvato, non fare nulla
                if(newLng.equals(MyPrefMgr.getPrefLng(SettingsActivity.this))) return;
                // Salva il nuovo codice lingua
                MyPrefMgr.setPrefLng(SettingsActivity.this, newLng);
                // Applica subito la lingua
                LocaleListCompat appLocale = LocaleListCompat.forLanguageTags(newLng);
                AppCompatDelegate.setApplicationLocales(appLocale);
            }

            @Override
            public void onNothingSelected(AdapterView<?> adapterView) { }
        });
    }


    private void setActivityResult() {
        mArlExp = registerForActivityResult(new ActivityResultContracts.StartActivityForResult(), new ActivityResultCallback<ActivityResult>() {
            @Override
            public void onActivityResult(ActivityResult result) {
                if (result.getResultCode() == SettingsActivity.RESULT_OK && result.getData() != null) {
                    Uri uri = result.getData().getData();
                    if (uri != null) startExportToUri(uri);
                }
            }
        });

        mArlImp = registerForActivityResult(new ActivityResultContracts.StartActivityForResult(), new ActivityResultCallback<ActivityResult>() {
            @Override
            public void onActivityResult(ActivityResult result) {
                if (result.getResultCode() == SettingsActivity.RESULT_OK && result.getData() != null) {
                    Uri uri = result.getData().getData();
                    String fileName = queryName(SettingsActivity.this, uri);
                    if (uri != null && fileName != null && fileName.endsWith(BACKUP_EXTENSION)) startImportFromUri(uri);
                    else {
                        Snackbar.make(mSetView, R.string.file_non_riconosciuto, Snackbar.LENGTH_LONG).show();
                        mPrbLoad.setVisibility(View.GONE);
                    }
                }
            }
        });
    }


    private void setBackupListener(){
        ImageButton imbExp = findViewById(R.id.imb_set_exp);
        imbExp.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                Intent itnExp = new Intent(Intent.ACTION_CREATE_DOCUMENT);
                itnExp.addCategory(Intent.CATEGORY_OPENABLE);
                itnExp.setType("*/*");
                itnExp.putExtra(Intent.EXTRA_TITLE,
                        "laundryNotes_db_"+ getTimeStamp(null) + BACKUP_EXTENSION);
                mArlExp.launch(itnExp);
            }
        });

        ImageButton imbImp = findViewById(R.id.imb_set_imp);
        imbImp.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                confermaImport(new Runnable() {
                    @Override
                    public void run() {
                        Intent itnImp = new Intent(Intent.ACTION_OPEN_DOCUMENT);
                        itnImp.addCategory(Intent.CATEGORY_OPENABLE);
                        itnImp.setType("*/*");
                        mArlImp.launch(itnImp);
                    }
                });
            }
        });
    }

    private String getTimeStamp(@Nullable LocalDateTime time){
        if(time == null) time = LocalDateTime.now();
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss");
        return time.format(formatter);
    }


    private void startExportToUri(Uri uri) {
        mPrbLoad.setVisibility(View.VISIBLE);
        mNoBackCallback.setEnabled(true);
        BackupDBManager bkpMgr = new BackupDBManager(this, LavanderiaDB.getInstance(this), Utilities.DB_NAME);
        bkpMgr.asyncExportDB(uri, new BackupDBManager.Callback() {
            @Override
            public void onComplete(Exception exc) {
                runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        if(exc == null) Snackbar.make(mSetView, R.string.database_esportato, Snackbar.LENGTH_LONG).show();
                        else if (exc.getCause() != null) Utilities.getCopyErrSnb(mSetView,
                                getString(R.string.impossibile_esportare), exc.getCause().getMessage()).show();
                        mPrbLoad.setVisibility(View.GONE);
                        mNoBackCallback.setEnabled(false);
                    }
                });
            }
        });
    }


    private void startImportFromUri(Uri uri) {
        mPrbLoad.setVisibility(View.VISIBLE);
        mNoBackCallback.setEnabled(true);
        BackupDBManager bkpMgr = new BackupDBManager(this, LavanderiaDB.getInstance(this), Utilities.DB_NAME);
        bkpMgr.asyncImportDB(uri, new BackupDBManager.Callback() {
            @Override
            public void onComplete(Exception exc) {
                runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        if (exc == null) {
                            Intent itnShowMainAct = new Intent(getApplicationContext(), MainActivity.class);
                            itnShowMainAct.putExtra("DB_AGGIORNATO", true);
                            itnShowMainAct.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK);
                            startActivity(itnShowMainAct);
                        }
                        else {
                            if (exc.getMessage() == null) return;
                            if (exc.getMessage().equals(BackupDBManager.EXC_NOT_VALID_FILE))
                                Snackbar.make(mSetView, R.string.file_non_valido, Snackbar.LENGTH_LONG).show();
                            else if (exc.getMessage().equals(BackupDBManager.EXC_CORRUPT_DB))
                                Snackbar.make(mSetView, R.string.file_non_riconosciuto, Snackbar.LENGTH_LONG).show();
                            else if (exc.getMessage().equals(BackupDBManager.EXC_GENERIC) && exc.getCause() != null) {
                                Utilities.getCopyErrSnb(mSetView, getString(R.string.impossibile_importare),
                                        exc.getCause().getMessage()).show();
                            }
                        }
                        mPrbLoad.setVisibility(View.GONE);
                        mNoBackCallback.setEnabled(false);
                    }
                });
            }
        });
    }


    private String queryName(Context context, Uri uri) {
        String result = null;
        Cursor cursor = context.getContentResolver().query(
                uri, new String[]{OpenableColumns.DISPLAY_NAME}, null, null, null
        );
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    result = cursor.getString(0);
                }
            } finally {
                cursor.close();
            }
        }
        return result;
    }

    private void confermaImport(Runnable onConfirmAction){
        new AlertDialog.Builder(this)
                .setTitle(R.string.conferma)
                .setMessage(getString(R.string.conferma_importazione))
                .setPositiveButton(R.string.si, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialogInterface, int i) {
                        onConfirmAction.run();
                    }
                })
                .setNegativeButton(R.string.no, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialogInterface, int i) {
                        dialogInterface.dismiss();
                    }
                }).show();
    }

    private void setBackPressed(){
        mNoBackCallback =  new OnBackPressedCallback(false) {
            @Override
            public void handleOnBackPressed() {
                Snackbar.make(mSetView, R.string.attendi, Snackbar.LENGTH_SHORT).show();
            }
        };
        getOnBackPressedDispatcher().addCallback(this, mNoBackCallback);
    }


}


