package org.eagsoftware.laundrynotes;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.content.res.AssetManager;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.content.res.TypedArray;
import android.graphics.Insets;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.TypedValue;
import android.view.View;
import android.view.Window;
import android.view.WindowInsets;

import androidx.appcompat.app.AppCompatDelegate;
import androidx.appcompat.content.res.AppCompatResources;
import androidx.core.os.LocaleListCompat;

import com.google.android.material.snackbar.Snackbar;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;

public class Utilities {

    public static final String DB_NAME = "lavanderia.db";


    /** @param xmlArrayResID: l'identificatore della risorsa "array" (es. R.array.resList)
     *  @return lista di interi corrispondenti agli id delle risorse puntate dall'array XML passato
     *          come argomento;
     *          -1 se la risorsa non esiste */
    public static List<Integer> getArrayListID(Context context, int xmlArrayResID){
        // Il try è necessario per essere sicuri di liberare le risorse anche in caso di errore
        TypedArray tarRes = context.getResources().obtainTypedArray(xmlArrayResID);
        List<Integer> res = new ArrayList<>();
        try{
            for (int i = 0; i < tarRes.length(); i++) res.add(tarRes.getResourceId(i, -1));
        }finally{
            tarRes.recycle();     // libera le risorse associate
        }
        return res;
    }

    public static Drawable getDrawableFromID(Context context, int drawableID){
        TypedValue tyvDrawable = new TypedValue();
        context.getTheme().resolveAttribute(drawableID, tyvDrawable, true);
        return AppCompatResources.getDrawable(context, tyvDrawable.resourceId);
    }

    public static int getIntColorFromID(Context context, int colorID){
        TypedValue tyvColor = new TypedValue();
        context.getTheme().resolveAttribute(colorID, tyvColor, true);
        return tyvColor.resourceId;
    }

    public static float getFloatFromID(Context context, int fractionID){
        TypedValue tyvFraction = new TypedValue();
        context.getResources().getValue(fractionID, tyvFraction, true);
        return tyvFraction.getFloat();
    }

    public static int getIntColorFromAttr(Context context, int attrID){
        TypedValue tyvAttr = new TypedValue();
        context.getTheme().resolveAttribute(attrID, tyvAttr, true);
        return tyvAttr.data;
    }


    public static void setLanguage(String lang){
        LocaleListCompat appLocale = LocaleListCompat.forLanguageTags(lang);
        AppCompatDelegate.setApplicationLocales(appLocale);
    }

    public static String getLanguage(Context context) {
        LocaleListCompat appLocales = AppCompatDelegate.getApplicationLocales();
        if(!appLocales.isEmpty()) {
            return Objects.requireNonNull(appLocales.get(0)).getLanguage();
        } else {
            // Fallback: lingua di sistema
            Locale systemLocale = context.getResources().getConfiguration().getLocales().get(0);
            return systemLocale.getLanguage();
        }
    }


    public static List<String> lingueDisponibiliInApp(Context context) {
        try {
            AssetManager am = context.getAssets();
            String[] locales = am.getLocales();
            LinkedHashSet<String> codes = new LinkedHashSet<>();

            if (locales == null || locales.length == 0) return new ArrayList<>();

            for (String loc : locales) {
                if (loc == null || loc.isEmpty()) continue;

                Locale locale = Locale.forLanguageTag(loc.replace('_', '-'));

                // Crea un contesto con quella configurazione
                Configuration cfg = new Configuration(context.getResources().getConfiguration());
                cfg.setLocales(new android.os.LocaleList(locale));
                Context cfgContext = context.createConfigurationContext(cfg);

                try {
                    // Legge il codice dal file strings.xml
                    String code = cfgContext.getString(R.string.app_language);
                    if (!code.isEmpty()) {
                        codes.add(code); // es. "en", "it", "fr"
                    }
                } catch (Resources.NotFoundException ex) {
                    // se manca la stringa in quella lingua, salta
                }
            }

            return new ArrayList<>(codes);
        } catch (Exception e) {
            throw new RuntimeException("Impossibile ottenere codici lingue", e);
        }
    }

    public static Snackbar getCopyErrSnb(View view, String msg, String error) {
        return Snackbar.make(view, msg, Snackbar.LENGTH_LONG)
                .setAction(R.string.copia_errore, new View.OnClickListener() {
                    @Override
                    public void onClick(View view) {
                        ClipboardManager clpMgr =
                                (ClipboardManager) view.getContext().getSystemService(Context.CLIPBOARD_SERVICE);
                        ClipData clipData = ClipData.newPlainText("Errore", error);
                        clpMgr.setPrimaryClip(clipData);

                        Snackbar.make(view, R.string.errore_copiato, Snackbar.LENGTH_LONG).show();
                    }
                });
    }


    public static String extractStackTrace(StackTraceElement[] stackTrace){
        StringBuilder msg = new StringBuilder();
        for (StackTraceElement ste : stackTrace) {
            msg.append(ste.toString()).append("\n");
        }
        return msg.toString();
    }


    public static void setSystemBarsAppearance(Window window, int color) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.VANILLA_ICE_CREAM) { // Android 15+
            window.getDecorView().setOnApplyWindowInsetsListener((view, insets) -> {
                Insets sysBarIns = insets.getInsets(WindowInsets.Type.systemBars());
                view.setBackgroundColor(color);
                // Applica padding sia sopra che sotto
                view.setPadding(
                        view.getPaddingLeft(),
                        sysBarIns.top,              // spazio per la status bar
                        view.getPaddingRight(),
                        sysBarIns.bottom            // spazio per la navigation bar
                );
                return insets;
            });
        } else {
            // For Android 14 and below
            window.setStatusBarColor(color);
        }
    }

    public static void copyStream(InputStream inputStream, OutputStream outputStream) throws IOException {
        byte[] buffer = new byte[8192];
        int len;
        while ((len = inputStream.read(buffer)) != -1) outputStream.write(buffer, 0, len);
        outputStream.flush();
    }

}
