package org.codeberg.quecomet.oshi.ui.screens.upload.components

import android.text.format.DateUtils
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material.icons.filled.DeleteForever
import androidx.compose.material.icons.filled.Refresh
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.work.WorkInfo
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.UploadWorkRepositoryMock
import org.codeberg.quecomet.oshi.data.room.UploadWorkAndOshiInstance
import org.codeberg.quecomet.oshi.ui.components.AnimatedLinearProgressIndicator
import org.codeberg.quecomet.oshi.ui.components.Card
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme

@Composable
fun UploadWorkItem(
    modifier: Modifier = Modifier,
    uploadWorkAndOshiInstance: UploadWorkAndOshiInstance,
    workInfoFlow: Flow<WorkInfo?>,
    onCancel: (workUUID: String) -> Unit,
    onRetry: (workUUID: String) -> Unit,
    onChangeInstanceRequest: (workUUID: String) -> Unit,
    onDelete: (workUUID: String) -> Unit,
) {
  val workInfo = workInfoFlow.collectAsStateWithLifecycle(initialValue = null)
  val isUploading =
      workInfo.value?.state == WorkInfo.State.RUNNING ||
          workInfo.value?.state == WorkInfo.State.ENQUEUED
  val workProgress = workInfo.value?.progress?.getFloat("progress", 0f) ?: 0f
  val workError = workInfo.value?.outputData?.getString("exception")

  Card(modifier) {
    Column(
        modifier =
            Modifier.padding(
                vertical = dimensionResource(R.dimen.padding_md),
                horizontal = dimensionResource(R.dimen.padding_lg),
            ),
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
      Row(
          modifier = Modifier.fillMaxWidth(),
          verticalAlignment = Alignment.CenterVertically,
      ) {
        Column(verticalArrangement = Arrangement.Bottom, modifier = Modifier.weight(1f)) {
          Text(
              text = uploadWorkAndOshiInstance.uploadWork.filename,
              maxLines = 1,
              overflow = TextOverflow.Ellipsis,
              modifier = Modifier.fillMaxWidth(),
          )
          Text(
              text =
                  stringResource(
                      R.string.created_at_relative_timestamp,
                      DateUtils.getRelativeTimeSpanString(
                              uploadWorkAndOshiInstance.uploadWork.createdAt.time,
                              System.currentTimeMillis(),
                              0L,
                              DateUtils.FORMAT_ABBREV_ALL,
                          )
                          .toString()),
              fontSize = 13.sp,
          )
        }

        Row {
          if (workInfo.value != null) {
            if (uploadWorkAndOshiInstance.uploadWork.canRetry && !isUploading) {
              IconButton(onClick = { onRetry(uploadWorkAndOshiInstance.uploadWork.workUUID) }) {
                Icon(
                    imageVector = Icons.Default.Refresh,
                    contentDescription = stringResource(R.string.retry))
              }
            }
          }
          IconButton(
              onClick = {
                if (isUploading) {
                  onCancel(uploadWorkAndOshiInstance.uploadWork.workUUID)
                } else {
                  onDelete(uploadWorkAndOshiInstance.uploadWork.workUUID)
                }
              },
          ) {
            if (isUploading) {
              Icon(
                  imageVector = Icons.Default.Close,
                  contentDescription = stringResource(R.string.cancel))
            } else {
              Icon(
                  imageVector = Icons.Default.DeleteForever,
                  contentDescription = stringResource(R.string.cancel))
            }
          }
        }
      }

      Row(
          horizontalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_md)),
          verticalAlignment = Alignment.CenterVertically,
          modifier = Modifier.fillMaxWidth(),
      ) {
        Text(
            text =
                stringResource(
                    R.string.uploading_to, uploadWorkAndOshiInstance.oshiInstance.toString()),
            fontSize = 13.sp,
            maxLines = 1,
            overflow = TextOverflow.Ellipsis,
            modifier = Modifier.weight(1f, true),
            color = MaterialTheme.colorScheme.onSurfaceVariant,
        )
        if (isUploading) {
          Text(
              text = "%.1f%%".format(workProgress * 100),
              fontSize = 13.sp,
          )
        } else {
          OutlinedButton(
              onClick = { onChangeInstanceRequest(uploadWorkAndOshiInstance.uploadWork.workUUID) },
              contentPadding =
              PaddingValues(
                  start = 7.dp,
                  top = 3.dp,
                  end = 7.dp,
                  bottom = 3.dp,
              ),
              modifier = Modifier.height(28.dp),
          ) {
            Text(
                stringResource(R.string.change_instance),
                fontSize = 13.sp,
                maxLines = 1,
                color = MaterialTheme.colorScheme.onSurfaceVariant,
            )
          }
        }
      }

      if (workInfo.value != null && !isUploading) {
        if (workInfo.value?.state == WorkInfo.State.CANCELLED) {
          Text(
              stringResource(R.string.upload_cancelled),
              color = MaterialTheme.colorScheme.onSurfaceVariant,
              textAlign = TextAlign.Center,
              modifier = Modifier.fillMaxWidth(),
              fontSize = 13.sp,
          )
        } else if (!uploadWorkAndOshiInstance.uploadWork.canRetry) {
          Text(
              if (workError == null) stringResource(R.string.cannot_retry)
              else stringResource(R.string.cannot_retry_with_reason, workError),
              color = MaterialTheme.colorScheme.error,
              textAlign = TextAlign.Center,
              modifier = Modifier.fillMaxWidth(),
              fontSize = 13.sp,
          )
        } else {
          Text(
              stringResource(R.string.upload_failed),
              color = MaterialTheme.colorScheme.error,
              fontSize = 13.sp,
          )
        }
      }

      if (workInfo.value != null && isUploading) {
        Spacer(Modifier.height(5.dp))
        AnimatedLinearProgressIndicator(workProgress, workProgress.compareTo(0f) == 0)
      }
    }
  }
}

@Preview("default")
@Composable
fun UploadItemPreview() {
  OshiTheme(darkTheme = true) {
    UploadWorkItem(
        uploadWorkAndOshiInstance = UploadWorkRepositoryMock.worksAndInstances[0],
        workInfoFlow = flow { emit(null) },
        onCancel = {},
        onRetry = {},
        onDelete = {},
        onChangeInstanceRequest = {},
    )
  }
}

@Preview("default")
@Composable
fun UploadItemNoRetryPreview() {
  OshiTheme(darkTheme = true) {
    UploadWorkItem(
        uploadWorkAndOshiInstance =
            UploadWorkRepositoryMock.worksAndInstances[0].copy(
                uploadWork =
                    UploadWorkRepositoryMock.worksAndInstances[0]
                        .uploadWork
                        .copy(canRetry = false)),
        workInfoFlow = flow { emit(null) },
        onCancel = {},
        onRetry = {},
        onDelete = {},
        onChangeInstanceRequest = {},
    )
  }
}
