package org.codeberg.quecomet.oshi.ui.screens.upload.components

import android.content.Intent
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyRow
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material.icons.filled.Upload
import androidx.compose.material3.Button
import androidx.compose.material3.FilledIconButton
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedIconButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.pluralStringResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import kotlinx.collections.immutable.PersistentList
import kotlinx.collections.immutable.persistentListOf
import kotlinx.collections.immutable.toPersistentList
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.UploadWorkRepositoryMock
import org.codeberg.quecomet.oshi.data.room.PreparedUploadWork
import org.codeberg.quecomet.oshi.data.room.toPreparedUploadWork
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.horizontalFadingEdge
import org.codeberg.quecomet.oshi.utils.getFileName

@Composable
fun FileSelector(
    selectedFiles: PersistentList<PreparedUploadWork>,
    onFilesSelected: (List<PreparedUploadWork>) -> Unit = {},
    onClearSelection: () -> Unit = {},
    onUploadRequested: () -> Unit = {},
) {
  val context = LocalContext.current

  val launcher =
      rememberLauncherForActivityResult(ActivityResultContracts.OpenMultipleDocuments()) { uriList
        ->
        if (uriList.isNotEmpty()) {
          onFilesSelected(
              uriList.map {
                try {
                  context.contentResolver.takePersistableUriPermission(
                      it,
                      Intent.FLAG_GRANT_READ_URI_PERMISSION or
                          Intent.FLAG_GRANT_WRITE_URI_PERMISSION)
                } catch (e: Throwable) {
                  // ignore
                }
                PreparedUploadWork(
                    filename = getFileName(it, context),
                    fileUri = it.toString(),
                    oshiInstanceId = 0,
                )
              })
        }
      }

  val listState = rememberLazyListState()

  Row(
      horizontalArrangement = Arrangement.SpaceBetween,
      verticalAlignment = Alignment.CenterVertically,
      modifier = Modifier.fillMaxWidth(),
  ) {
    if (selectedFiles.isEmpty()) {
      Button(onClick = { launcher.launch(arrayOf("*/*")) }) {
        Text(stringResource(R.string.select_files))
      }
    } else {
      Row() {
        FilledIconButton(onClick = { onUploadRequested() }) {
          Icon(
              imageVector = Icons.Default.Upload,
              contentDescription = stringResource(R.string.upload))
        }
        OutlinedIconButton(
            onClick = { onClearSelection() },
            border = BorderStroke(1.dp, MaterialTheme.colorScheme.error),
        ) {
          Icon(
              imageVector = Icons.Default.Close,
              contentDescription = stringResource(R.string.clear_selected_files),
              tint = MaterialTheme.colorScheme.error,
          )
        }
      }
    }
    Box(
        modifier =
            Modifier.weight(1f, true).padding(horizontal = dimensionResource(R.dimen.page_padding)),
    ) {
      if (selectedFiles.isEmpty()) {
        Text(
            stringResource(R.string.no_file_selected),
            textAlign = TextAlign.Center,
            modifier = Modifier.fillMaxWidth(),
        )
      } else {
        Column(
            horizontalAlignment =
                if (selectedFiles.size == 1) Alignment.CenterHorizontally else Alignment.Start,
            modifier = Modifier.fillMaxWidth(),
        ) {
          Text(
              pluralStringResource(
                  R.plurals.number_of_selected_files, selectedFiles.size, selectedFiles.size),
              style = MaterialTheme.typography.labelSmall)
          LazyRow(
              state = listState,
              horizontalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.page_padding)),
              verticalAlignment = Alignment.CenterVertically,
              modifier =
                  Modifier.weight(1f, false)
                      .horizontalFadingEdge(
                          listState, 100.dp, edgeColor = MaterialTheme.colorScheme.background)) {
                items(items = selectedFiles, key = { it.fileUri }) { Text("\"${it.filename}\"") }
              }
        }
      }
    }
  }
}

@Preview("empty")
@Composable
fun FileSelectorEmptyPreview() {
  OshiTheme() { FileSelector(persistentListOf()) {} }
}

@Preview("with items")
@Composable
fun FileSelectorPreview() {
  OshiTheme() {
    FileSelector(
        UploadWorkRepositoryMock.works.map { it.toPreparedUploadWork() }.toPersistentList(),
    ) {}
  }
}
