package org.codeberg.quecomet.oshi.ui.screens.upload.components

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Button
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.SnackbarDuration
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.OshiInstanceRepositoryMock
import org.codeberg.quecomet.oshi.data.room.OshiInstance
import org.codeberg.quecomet.oshi.model.SnackbarMessage
import org.codeberg.quecomet.oshi.model.UserMessage
import org.codeberg.quecomet.oshi.ui.components.BasicDialog
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ConfirmFingerprintDialog(
    modifier: Modifier = Modifier,
    show: Boolean = true,
    oshiInstance: OshiInstance? = null,
    onConfirm: suspend (OshiInstance) -> Unit,
    onDismissRequest: () -> Unit,
) {
  val scope = rememberCoroutineScope()
  if (show) {
    var working by remember { mutableStateOf(false) }
    BasicDialog(
        onDismissRequest = onDismissRequest,
    ) {
      Column(
          horizontalAlignment = Alignment.CenterHorizontally,
          verticalArrangement = Arrangement.spacedBy(5.dp),
          modifier = modifier.padding(dimensionResource(R.dimen.padding_xl)),
      ) {
        if (oshiInstance != null) {
          Text(
              text = stringResource(R.string.title_confirm_fingerprint),
              style = MaterialTheme.typography.titleLarge,
          )

          Spacer(Modifier)

          Text(text = "${oshiInstance.protocol.name}://${oshiInstance.host}")

          Spacer(Modifier)

          OutlinedTextField(
              modifier = Modifier.fillMaxWidth(),
              value = oshiInstance.sha1Fingerprint ?: "error",
              onValueChange = {},
              textStyle = MaterialTheme.typography.bodyMedium,
              readOnly = true,
              maxLines = 2,
              label = { Text(stringResource(R.string.instance_ssl_sha1_fingerprint)) },
          )

          OutlinedTextField(
              modifier = Modifier.fillMaxWidth(),
              value = oshiInstance.sha256Fingerprint ?: "error",
              onValueChange = {},
              textStyle = MaterialTheme.typography.bodyMedium,
              readOnly = true,
              maxLines = 3,
              label = { Text(stringResource(R.string.instance_ssl_sha256_fingerprint)) },
          )

          Row(
              horizontalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_md)),
          ) {
            TextButton(onClick = { onDismissRequest() }) { Text(stringResource(R.string.cancel)) }
            Button(
                onClick = {
                  if (working) return@Button
                  working = true
                  scope
                      .launch { onConfirm(oshiInstance) }
                      .invokeOnCompletion {
                        if (it != null) {
                          SnackbarManager.showMessage(
                              SnackbarMessage.from(
                                  UserMessage.from(
                                      R.string.error_while_saving, it.message ?: it.toString()),
                                  duration = SnackbarDuration.Short,
                                  withDismissAction = true,
                              ))
                        }
                        onDismissRequest()
                        working = false
                      }
                },
                enabled = !working,
            ) {
              if (working) {
                CircularProgressIndicator()
              } else {
                Text(stringResource(R.string.label_confirm))
              }
            }
          }
        } else {
          CircularProgressIndicator()
        }
      }
    }
  }
}

@Preview("Confirm Dialog")
@Composable
fun ConfirmFingerprintDialogPreview() {
  OshiTheme {
    Surface {
      ConfirmFingerprintDialog(
          oshiInstance = OshiInstanceRepositoryMock.instances[2],
          onConfirm = {},
          onDismissRequest = {},
      )
    }
  }
}
