package org.codeberg.quecomet.oshi.ui.screens.upload

import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.util.Log
import androidx.compose.material3.SnackbarDuration
import androidx.compose.material3.SnackbarResult
import androidx.compose.runtime.Stable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.net.toUri
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import androidx.paging.PagingData
import androidx.paging.cachedIn
import androidx.work.WorkInfo
import dagger.hilt.android.lifecycle.HiltViewModel
import io.ktor.client.request.get
import io.ktor.http.URLProtocol
import kotlinx.collections.immutable.PersistentList
import kotlinx.collections.immutable.persistentListOf
import kotlinx.collections.immutable.toPersistentList
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.ShareIntentHandler
import org.codeberg.quecomet.oshi.createFileInCacheAndGetUri
import org.codeberg.quecomet.oshi.data.OshiInstanceRepository
import org.codeberg.quecomet.oshi.data.UploadWorkRepository
import org.codeberg.quecomet.oshi.data.UserSettingsRepository
import org.codeberg.quecomet.oshi.data.room.OshiInstance
import org.codeberg.quecomet.oshi.data.room.PreparedUploadWork
import org.codeberg.quecomet.oshi.data.room.UploadWorkAndOshiInstance
import org.codeberg.quecomet.oshi.model.SnackbarMessage
import org.codeberg.quecomet.oshi.model.UserMessage
import org.codeberg.quecomet.oshi.network.FingerprintTrustManager
import org.codeberg.quecomet.oshi.network.createHttpClient
import org.codeberg.quecomet.oshi.network.exceptions.FingerprintDetectedException
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.utils.getFileName
import org.codeberg.quecomet.oshi.utils.parcelable
import org.codeberg.quecomet.oshi.utils.parcelableArrayList
import org.codeberg.quecomet.oshi.utils.removeUploadProgressNotification
import java.io.IOException
import java.security.cert.CertificateException
import java.util.UUID
import java.util.concurrent.CancellationException
import javax.inject.Inject
import javax.net.ssl.SSLHandshakeException

private const val TAG = "UploadViewModel"

@HiltViewModel
@Stable
class UploadViewModel
@Inject
constructor(
    private val savedStateHandle: SavedStateHandle,
    private val uploadWorkRepository: UploadWorkRepository,
    private val oshiInstanceRepository: OshiInstanceRepository,
    private val userSettingsRepository: UserSettingsRepository,
    val shareIntentHandler: ShareIntentHandler,
) : ViewModel() {

  val uploadWorkPagingDataFlow: Flow<PagingData<UploadWorkAndOshiInstance>> =
      uploadWorkRepository.getUploadWorkAndOshiInstanceList().cachedIn(viewModelScope)

  var oshiInstanceSearchTerm = MutableStateFlow("")
    private set

  @OptIn(ExperimentalCoroutinesApi::class)
  val oshiInstancesPagingDataFlow: Flow<PagingData<OshiInstance>> =
      oshiInstanceSearchTerm.flatMapLatest { term ->
        oshiInstanceRepository.getInstanceList(term).cachedIn(viewModelScope)
      }

  val userSettings =
      userSettingsRepository.userSettingsFlow.stateIn(
          scope = viewModelScope,
          started = SharingStarted.WhileSubscribed(5_000),
          initialValue = null)

  val userSelectedOshiInstance =
      userSettingsRepository.userSettingsFlow
          .map { oshiInstanceRepository.getInstance(it.selectedOshiInstanceId) }
          .stateIn(
              scope = viewModelScope,
              started = SharingStarted.WhileSubscribed(5_000),
              initialValue = null)

  var instanceToConfirmSSLFingerprint by mutableStateOf<OshiInstance?>(null)
    private set

  var checkingFingerprint by mutableStateOf(false)
    private set

  var showNotificationPermissionRequest by mutableStateOf(false)
    private set

  var selectedFiles by mutableStateOf<PersistentList<PreparedUploadWork>>(persistentListOf())
    private set

  var workUUIDRequestingInstanceChange by mutableStateOf("")
    private set

  fun updateOshiInstanceSearchTerm(value: String) {
    oshiInstanceSearchTerm.value = value
  }

  fun updateInstanceToConfirmSSLFingerprint(oshiInstance: OshiInstance?) {
    instanceToConfirmSSLFingerprint = oshiInstance
  }

  fun updateSelectedFiles(value: List<PreparedUploadWork>) {
    selectedFiles = value.toPersistentList()
    viewModelScope.launch {
      val settings = userSettings.firstOrNull()
      if (settings != null && !settings.hasShownNotificationPermissionRequest) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
          updateShowNotificationPermissionRequest(true)
        }

        userSettingsRepository.setHasShownNotificationPermissionRequest(true)
      }
    }
  }

  fun updateShowNotificationPermissionRequest(value: Boolean) {
    showNotificationPermissionRequest = value
  }

  fun updateUploadDestroyAfterDL(value: Boolean) {
    viewModelScope.launch { userSettingsRepository.setUploadDestroyAfterDL(value) }
  }

  fun updateUploadRandomizeName(value: Boolean) {
    viewModelScope.launch { userSettingsRepository.setUploadRandomizeName(value) }
  }

  fun updateUploadShortenUrl(value: Boolean) {
    viewModelScope.launch { userSettingsRepository.setUploadShortenUrl(value) }
  }

  fun updateUploadExpiration(value: Int) {
    viewModelScope.launch { userSettingsRepository.setUploadExpiration(value) }
  }

  fun getUploadWorkWorkInfo(workUUID: String?): Flow<WorkInfo?> {
    return uploadWorkRepository.getUploadWorkWorkInfo(workUUID)
  }

  suspend fun onDeleteOshiInstance(oshiInstance: OshiInstance) {
    oshiInstanceRepository.delete(oshiInstance.id)
    if (userSettingsRepository.fetchSettings().selectedOshiInstanceId == oshiInstance.id) {
      userSettingsRepository.setSelectedOshiInstance(-1)
    }
  }

  var confirmFingerprintJob: Job? = null

  fun confirmFingerprint(oshiInstance: OshiInstance?) {
    confirmFingerprintJob?.cancel()
    confirmFingerprintJob =
        viewModelScope.launch {
          if (oshiInstance != null &&
              oshiInstance.protocol == URLProtocol.HTTPS &&
              (oshiInstance.sha1Fingerprint.isNullOrBlank() ||
                  !oshiInstance.fingerprintConfirmedByUser)) {

            if (!oshiInstance.sha1Fingerprint.isNullOrBlank()) {
              instanceToConfirmSSLFingerprint = oshiInstance
            } else {
              checkingFingerprint = true
              val httpClient =
                  createHttpClient(
                      userSettingsRepository.fetchSettings(),
                      oshiInstance,
                      FingerprintTrustManager())
              try {
                httpClient.get {
                  url {
                    protocol = URLProtocol.HTTPS
                    host = oshiInstance.host
                  }
                }
              } catch (e: Throwable) {
                if (e is SSLHandshakeException &&
                    e.cause is CertificateException &&
                    e.cause?.cause is FingerprintDetectedException) {
                  val fpd: FingerprintDetectedException =
                      e.cause!!.cause as FingerprintDetectedException
                  instanceToConfirmSSLFingerprint =
                      oshiInstance.copy(
                          sha1Fingerprint = fpd.sha1Fingerprint,
                          sha256Fingerprint = fpd.sha256Fingerprint,
                          fingerprintConfirmedByUser = true)
                } else {
                  if (e.message?.contains("user_cancelled_job") == true) {
                    SnackbarManager.showMessage(
                        SnackbarMessage.from(
                            UserMessage.from(R.string.snack_notice_fingerprint_cancel),
                            actionLabelMessage = UserMessage.from(R.string.retry),
                            duration = SnackbarDuration.Long,
                            withDismissAction = true,
                            onSnackbarResult = {
                              if (it == SnackbarResult.ActionPerformed) {
                                confirmFingerprint(oshiInstance)
                              }
                            },
                        ),
                    )
                  } else {
                    SnackbarManager.showMessage(
                        SnackbarMessage.from(
                            UserMessage.from(
                                R.string.snack_failed_to_retrieve_instance_fingerprint),
                            actionLabelMessage = UserMessage.from(R.string.retry),
                            duration = SnackbarDuration.Long,
                            withDismissAction = true,
                            onSnackbarResult = {
                              if (it == SnackbarResult.ActionPerformed) {
                                confirmFingerprint(oshiInstance)
                              }
                            },
                        ),
                    )
                  }
                }
              } finally {
                checkingFingerprint = false
              }
            }
          }
        }
  }

  suspend fun onInstanceSSLConfirmed(instance: OshiInstance) {
    oshiInstanceRepository.updateInstance(instance)
  }

  fun setSelectedOshiInstance(instance: OshiInstance) {
    if (workUUIDRequestingInstanceChange.isNotBlank()) {
      viewModelScope.launch {
        val workUUID = workUUIDRequestingInstanceChange
        workUUIDRequestingInstanceChange = ""
        uploadWorkRepository.changeInstance(workUUID, instance.id)
      }
    } else {
      viewModelScope.launch { userSettingsRepository.setSelectedOshiInstance(instance.id) }
    }
  }

  suspend fun saveInstance(it: OshiInstance, isEdit: Boolean) {
    if (isEdit) {
      oshiInstanceRepository.updateInstance(it)
    } else {
      oshiInstanceRepository.addInstance(it)
    }
  }

  fun uploadSelectedFiles() {
    val settings = userSettings.value ?: return
    val filesToUpload = selectedFiles
    selectedFiles = persistentListOf()
    viewModelScope.launch {
      filesToUpload.forEach {
        uploadWorkRepository.createWork(
            it.copy(
                oshiInstanceId = settings.selectedOshiInstanceId,
                destroyAfterDl = settings.uploadDestroyAfterDl,
                randomizeFilename = settings.uploadRandomizeName,
                shortenUrl = settings.uploadShortenUrl,
                expireInMinutes = settings.uploadExpiration,
            ))
      }
    }
  }

  fun retryUploadWork(workUUID: String) {
    uploadWorkRepository.retry(workUUID)
  }

  fun cancelUploadWork(workUUID: String) {
    uploadWorkRepository.cancel(workUUID)
  }

  fun deleteUploadWork(workUUID: String, context: Context? = null) {
    if (context != null) {
      removeUploadProgressNotification(workUUID, context)
    }
    viewModelScope.launch { uploadWorkRepository.delete(workUUID) }
  }

  private fun createTextFileAndReturnUri(context: Context, text: String): Uri? {
    return try {
      return "${context.createFileInCacheAndGetUri(text, UUID.randomUUID().toString() + ".txt")}?tempfile"
          .toUri()
    } catch (e: IOException) {
      Log.d(TAG, "createTextFileAndReturnUri: " + (e.message ?: e.toString()))
      null
    }
  }

  fun handleShareIntent(intent: Intent, context: Context) {
    viewModelScope.launch {
      val uriList: List<Uri> =
          when (intent.action) {
            Intent.ACTION_SEND -> {
              intent.parcelable<Uri>(Intent.EXTRA_STREAM)?.let { listOf(it) }
                  ?: intent.getCharSequenceExtra(Intent.EXTRA_TEXT)?.toString()?.let { text ->
                    createTextFileAndReturnUri(context, text)?.let { listOf(it) }
                  }
                  ?: emptyList()
            }
            Intent.ACTION_SEND_MULTIPLE -> {
              intent.parcelableArrayList<Uri>(Intent.EXTRA_STREAM) ?: emptyList()
            }
            else -> emptyList()
          }

      selectedFiles =
          uriList
              .map {
                try {
                  context.contentResolver.takePersistableUriPermission(
                      it,
                      Intent.FLAG_GRANT_READ_URI_PERMISSION or
                          Intent.FLAG_GRANT_WRITE_URI_PERMISSION)
                } catch (e: Throwable) {
                  // ignore
                }
                PreparedUploadWork(
                    filename = getFileName(it, context),
                    fileUri = it.toString(),
                    oshiInstanceId = 0,
                )
              }
              .toPersistentList()
    }
  }

  fun cancelFingerprintCheck() {
    confirmFingerprintJob?.cancel(CancellationException("user_cancelled_job"))
  }

  fun onChangeInstanceRequest(workUUID: String) {
    workUUIDRequestingInstanceChange = workUUID
  }

  fun onChangeInstanceRequestCancelled() {
    workUUIDRequestingInstanceChange = ""
  }
}
