package org.codeberg.quecomet.oshi.ui.screens.settings

import androidx.compose.material3.SnackbarDuration
import androidx.compose.runtime.Stable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.hilt.android.lifecycle.HiltViewModel
import org.codeberg.quecomet.oshi.PreferredProxyType
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.SSLCheckMode
import org.codeberg.quecomet.oshi.copy
import org.codeberg.quecomet.oshi.data.UserSettingsRepository
import org.codeberg.quecomet.oshi.model.SnackbarMessage
import org.codeberg.quecomet.oshi.model.UserMessage
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
@Stable
class SettingsViewModel
@Inject
constructor(
  private val savedStateHandle: SavedStateHandle,
  private val userSettingsRepository: UserSettingsRepository,
) : ViewModel() {

  var sslCheckModeDraft by mutableStateOf(SSLCheckMode.CERTIFICATE_ONLY)
    private set

  var isProxyEnabledDraft by mutableStateOf(false)
    private set

  var proxyHostDraft by mutableStateOf("")
    private set

  var proxyPortDraft by mutableStateOf("")
    private set

  var preferredProxyTypeDraft by mutableStateOf(PreferredProxyType.SOCKS5)
    private set

  fun updateSSLCheckModeDraft(value: SSLCheckMode) {
    sslCheckModeDraft = value
  }

  fun updateIsProxyEnabledDraft(value: Boolean) {
    isProxyEnabledDraft = value
  }

  fun updateProxyHostDraft(value: String) {
    proxyHostDraft = value
  }

  fun updateProxyPortDraft(value: String) {
    proxyPortDraft = value
  }

  fun updatePreferredProxyTypeDraft(value: PreferredProxyType) {
    preferredProxyTypeDraft = value
  }

  fun reloadSettings() {
    viewModelScope.launch {
      userSettingsRepository.userSettingsFlow.firstOrNull()?.let {
        sslCheckModeDraft = it.sslCheckMode
        isProxyEnabledDraft = it.isProxyEnabled
        proxyHostDraft = it.proxyHost
        proxyPortDraft = it.proxyPort
        preferredProxyTypeDraft = it.preferredProxyType
      }
    }
  }

  fun saveSSLCheckModeSettings() {
    viewModelScope.launch {
      userSettingsRepository.saveSettings(
          userSettingsRepository.fetchSettings().copy {
            sslCheckMode = sslCheckModeDraft
          },
      )
      showSuccessSnack()
    }
  }

  fun saveProxySettings() {
    viewModelScope.launch {
      userSettingsRepository.saveSettings(
          userSettingsRepository.fetchSettings().copy {
            isProxyEnabled = isProxyEnabledDraft
            proxyHost = proxyHostDraft
            proxyPort = proxyPortDraft
            preferredProxyType = preferredProxyTypeDraft
          },
      )
      showSuccessSnack()
    }
  }

  private fun showSuccessSnack() {
    SnackbarManager.showMessage(
        SnackbarMessage.from(
            UserMessage.from(R.string.snack_saved_changes_successfully),
            duration = SnackbarDuration.Short,
            withDismissAction = true,
        ),
    )
  }
}
