package org.codeberg.quecomet.oshi.ui.screens.managefile

import android.icu.text.SimpleDateFormat
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.widthIn
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.selection.SelectionContainer
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.ContentCopy
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.UploadedFileRepositoryMock
import org.codeberg.quecomet.oshi.ui.components.Card
import org.codeberg.quecomet.oshi.ui.components.ConfirmDeleteDialog
import org.codeberg.quecomet.oshi.ui.components.RefreshButton
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.ui.navigation.NavArgs
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.humanReadableByteCountBin
import org.codeberg.quecomet.oshi.ui.utils.noRippleClickable
import org.codeberg.quecomet.oshi.ui.utils.verticalFadingEdge
import org.codeberg.quecomet.oshi.utils.copyToClipboard
import org.codeberg.quecomet.oshi.utils.isWorking
import java.net.URLEncoder

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ManageFile(
    viewModel: ManageFileViewModel,
    upPress: () -> Unit,
    modifier: Modifier = Modifier,
) {
  val context = LocalContext.current
  val currentLocale = context.resources.configuration.locales.get(0)
  val uriHandler = LocalUriHandler.current
  val entity by viewModel.uploadedFileAndOshiInstance.collectAsStateWithLifecycle()
  val oshiInstance = entity?.oshiInstance
  val uploadedFile = entity?.uploadedFile
  val scrollState = rememberScrollState()
  val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())
  val focusManager = LocalFocusManager.current

  val simpleDateTimeFormatter = SimpleDateFormat("EEE MMM dd HH:mm:ss yyyy", currentLocale)

  if (uploadedFile?.deletedInRemote == false) {
    LaunchedEffect(true) {
      coroutineScope {
        while (isActive) {
          viewModel.refreshInfo()
          delay(20000L)
        }
      }
    }
  }

  Scaffold(
      modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
      snackbarHost = { SnackbarHost(hostState = SnackbarManager.snackbarHostState) },
      topBar = {
        CenterAlignedTopAppBar(
            scrollBehavior = scrollBehavior,
            colors =
                TopAppBarDefaults.centerAlignedTopAppBarColors(
                    scrolledContainerColor = MaterialTheme.colorScheme.inverseOnSurface,
                ),
            title = {
              Column(horizontalAlignment = Alignment.CenterHorizontally) {
                Text(stringResource(R.string.title_manage_file))
                if (!viewModel.fileDeleted) {
                  SelectionContainer {
                    val filename = uploadedFile?.originalFilename ?: ""
                    Text(
                        filename,
                        color = MaterialTheme.colorScheme.onSurfaceVariant,
                        fontSize = 16.sp,
                        maxLines = 1,
                        overflow = TextOverflow.Ellipsis,
                        modifier = Modifier.widthIn(0.dp, 200.dp),
                    )
                  }
                }
              }
            },
            navigationIcon = {
              IconButton(onClick = { upPress() }) {
                Icon(
                    modifier = Modifier.size(32.dp),
                    imageVector = Icons.AutoMirrored.Default.ArrowBack,
                    contentDescription = null,
                )
              }
            },
            actions = {
              RefreshButton(
                  onClick = { viewModel.refreshInfo() },
                  enabled = !viewModel.refreshWorkState.isWorking(),
                  refreshing = viewModel.refreshWorkState.isWorking(),
              )
            },
        )
      },
  ) { padding ->
    Column(
        modifier
            .fillMaxSize()
            .padding(padding)
            .verticalFadingEdge(scrollState, dimensionResource(R.dimen.fading_edge))
            .padding(dimensionResource(R.dimen.padding_lg))
            .verticalScroll(scrollState)
            .noRippleClickable { focusManager.clearFocus() },
        verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_md))) {
          if (oshiInstance != null && uploadedFile != null) {
            Row(
                verticalAlignment = Alignment.CenterVertically,
                modifier = Modifier.fillMaxWidth(),
            ) {
              Text(stringResource(R.string.link), modifier = Modifier.weight(0.8f, true))
              Row(
                  Modifier.weight(1f, true)
                      .padding(start = dimensionResource(R.dimen.padding_lg))) {
                    OutlinedButton(
                        onClick = {
                          copyToClipboard(
                              context,
                              "${oshiInstance.protocol.name}://${oshiInstance.host}${uploadedFile.downloadPath}")
                        },
                    ) {
                      Icon(Icons.Default.ContentCopy, null)
                      Spacer(Modifier.width(4.dp))
                      Text(stringResource(R.string.link))
                    }
                  }
            }
            if (!oshiInstance.alternateHost.isNullOrBlank()) {
              Row(
                  verticalAlignment = Alignment.CenterVertically,
                  modifier = Modifier.fillMaxWidth(),
              ) {
                Text(
                    stringResource(R.string.alternate_link), modifier = Modifier.weight(0.8f, true))
                Row(
                    Modifier.weight(1f, true)
                        .padding(start = dimensionResource(R.dimen.padding_lg))) {
                      OutlinedButton(
                          onClick = {
                            copyToClipboard(
                                context,
                                "${oshiInstance.alternateProtocol.name}://${oshiInstance.alternateHost}${uploadedFile.downloadPath}")
                          },
                      ) {
                        Icon(Icons.Default.ContentCopy, null)
                        Spacer(Modifier.width(4.dp))
                        Text(stringResource(R.string.alternate_link))
                      }
                    }
              }
            }
            Row(
                modifier = Modifier.fillMaxWidth(),
            ) {
              Text(
                  stringResource(R.string.filename_on_instance),
                  modifier = Modifier.weight(0.8f, true))
              SelectionContainer(
                  Modifier.weight(1f, true)
                      .padding(start = dimensionResource(R.dimen.padding_lg))) {
                    Text(uploadedFile.filename.ifBlank { "-" })
                  }
            }
            Row(
                modifier = Modifier.fillMaxWidth(),
            ) {
              Text(stringResource(R.string.created_at), modifier = Modifier.weight(0.8f, true))
              SelectionContainer(
                  Modifier.weight(1f, true)
                      .padding(start = dimensionResource(R.dimen.padding_lg))) {
                    Text(simpleDateTimeFormatter.format(uploadedFile.createdAt))
                  }
            }
            Row(
                modifier = Modifier.fillMaxWidth(),
            ) {
              Text(stringResource(R.string.expires_at), modifier = Modifier.weight(0.8f, true))
              SelectionContainer(
                  Modifier.weight(1f, true)
                      .padding(start = dimensionResource(R.dimen.padding_lg))) {
                    Text(simpleDateTimeFormatter.format(uploadedFile.expiresAt))
                  }
            }
            Row(
                modifier = Modifier.fillMaxWidth(),
            ) {
              Text(stringResource(R.string.size), modifier = Modifier.weight(0.8f, true))
              SelectionContainer(
                  Modifier.weight(1f, true)
                      .padding(start = dimensionResource(R.dimen.padding_lg))) {
                    Text(humanReadableByteCountBin(uploadedFile.size))
                  }
            }
            if (uploadedFile.mimeType.isNotBlank()) {
              Row(
                  modifier = Modifier.fillMaxWidth(),
              ) {
                Text(stringResource(R.string.type), modifier = Modifier.weight(0.8f, true))
                SelectionContainer(
                    Modifier.weight(1f, true)
                        .padding(start = dimensionResource(R.dimen.padding_lg))) {
                      Text(uploadedFile.mimeType)
                    }
              }
            }
            if (!uploadedFile.hashsumSha1.isNullOrBlank()) {
              Row(
                  modifier = Modifier.fillMaxWidth(),
              ) {
                Text(stringResource(R.string.hashsum_sha1), modifier = Modifier.weight(0.8f, true))
                SelectionContainer(
                    Modifier.weight(1f, true)
                        .padding(start = dimensionResource(R.dimen.padding_lg))) {
                      Text(uploadedFile.hashsumSha1)
                    }
              }
            }

            Row(
                verticalAlignment = Alignment.CenterVertically,
                modifier = Modifier.fillMaxWidth(),
            ) {
              Text(
                  stringResource(R.string.destroy_after_download),
                  modifier = Modifier.weight(0.8f, true))
              Row(
                  verticalAlignment = Alignment.CenterVertically,
                  horizontalArrangement = Arrangement.SpaceBetween,
                  modifier =
                      Modifier.weight(1f, true)
                          .padding(start = dimensionResource(R.dimen.padding_lg))) {
                    Text(
                        if (uploadedFile.destroyAfterDl) stringResource(R.string.yes)
                        else stringResource(R.string.no))
                    Spacer(Modifier.width(dimensionResource(R.dimen.padding_lg)))
                    OutlinedButton(
                        enabled = !viewModel.toggleDestroyWorkState.isWorking(),
                        onClick = { viewModel.toggleDestroyAfterDl() },
                    ) {
                      if (viewModel.toggleDestroyWorkState.isWorking()) {
                        CircularProgressIndicator(Modifier.size(18.dp))
                      } else {
                        Text(stringResource(R.string.toggle))
                      }
                    }
                  }
            }

            Row(
                verticalAlignment = Alignment.CenterVertically,
                modifier = Modifier.fillMaxWidth(),
            ) {
              Text(stringResource(R.string.onion_only), modifier = Modifier.weight(0.8f, true))
              Row(
                  verticalAlignment = Alignment.CenterVertically,
                  horizontalArrangement = Arrangement.SpaceBetween,
                  modifier =
                      Modifier.weight(1f, true)
                          .padding(start = dimensionResource(R.dimen.padding_lg))) {
                    Text(
                        if (uploadedFile.isOnionOnly) stringResource(R.string.yes)
                        else stringResource(R.string.no))
                    Spacer(Modifier.width(dimensionResource(R.dimen.padding_lg)))
                    OutlinedButton(
                        enabled = !viewModel.toggleOnionWorkState.isWorking(),
                        onClick = { viewModel.toggleOnionOnly() },
                    ) {
                      if (viewModel.toggleOnionWorkState.isWorking()) {
                        CircularProgressIndicator(Modifier.size(18.dp))
                      } else {
                        Text(stringResource(R.string.toggle))
                      }
                    }
                  }
            }

            if (!uploadedFile.hits.isNullOrBlank()) {
              Row(
                  modifier = Modifier.fillMaxWidth(),
              ) {
                Text(stringResource(R.string.hits), modifier = Modifier.weight(0.8f, true))
                SelectionContainer(
                    Modifier.weight(1f, true)
                        .padding(start = dimensionResource(R.dimen.padding_lg))) {
                      Text(uploadedFile.hits)
                    }
              }
            }
            Spacer(Modifier.height(dimensionResource(R.dimen.padding_lg)))
            Row(
                horizontalArrangement = Arrangement.Start,
                verticalAlignment = Alignment.CenterVertically,
                modifier = Modifier.fillMaxWidth(),
            ) {
              var showDeleteConfirmDialog by remember { mutableStateOf(false) }
              if (showDeleteConfirmDialog) {
                ConfirmDeleteDialog(
                    itemLabel = uploadedFile.originalFilename,
                    onConfirm = { viewModel.requestFileDelete { upPress() } },
                    workState = viewModel.requestFileDeleteWorkState,
                    onDismissRequest = { showDeleteConfirmDialog = false },
                )
              }

              Row(
                  modifier =
                      Modifier.weight(0.8f, true)
                          .padding(start = dimensionResource(R.dimen.padding_lg)),
                  horizontalArrangement = Arrangement.End) {
                    Button(
                        onClick = { showDeleteConfirmDialog = true },
                        colors =
                            ButtonDefaults.buttonColors(
                                containerColor = MaterialTheme.colorScheme.errorContainer,
                                contentColor = MaterialTheme.colorScheme.onErrorContainer,
                            ),
                    ) {
                      Text(stringResource(R.string.delete))
                    }
                  }
              Row(
                  modifier =
                      Modifier.weight(1f, true)
                          .padding(start = dimensionResource(R.dimen.padding_lg)),
              ) {
                OutlinedButton(
                    onClick = {
                      copyToClipboard(
                          context,
                          "${oshiInstance.protocol.name}://${oshiInstance.host}${uploadedFile.managePath}")
                    },
                ) {
                  Icon(Icons.Default.ContentCopy, null)
                  Spacer(Modifier.width(4.dp))
                  Text(stringResource(R.string.manage_link))
                }
              }
            }
          } else if (viewModel.fileDeleted) {
            Card {
              Column(
                  horizontalAlignment = Alignment.CenterHorizontally,
                  modifier =
                      Modifier.padding(
                          horizontal = dimensionResource(R.dimen.page_padding), vertical = 70.dp)) {
                    Text(stringResource(R.string.file_deleted))
                  }
            }
          } else {
            Card {
              Column(
                  horizontalAlignment = Alignment.CenterHorizontally,
                  modifier =
                      Modifier.padding(
                          horizontal = dimensionResource(R.dimen.page_padding), vertical = 70.dp)) {
                    Text(stringResource(R.string.unknown_error))
                  }
            }
          }
        }
  }
}

fun getMockViewModel() =
    ManageFileViewModel(
        SavedStateHandle().apply {
          this[NavArgs.MANAGE_FILE_PATH_URI_ENCODED] =
              URLEncoder.encode(UploadedFileRepositoryMock.files[0].managePath, "UTF-8")
        },
        UploadedFileRepositoryMock(),
    )

@Preview
@Composable
fun ManageFilePreview() {
  OshiTheme {
    ManageFile(
        viewModel = getMockViewModel(),
        upPress = {},
    )
  }
}
