package org.codeberg.quecomet.oshi.ui.screens.history

import android.text.format.DateUtils
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.widthIn
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ContentCopy
import androidx.compose.material.icons.filled.DeleteForever
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.OutlinedIconButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.UploadedFileRepositoryMock
import org.codeberg.quecomet.oshi.data.room.UploadedFile
import org.codeberg.quecomet.oshi.data.room.UploadedFileAndOshiInstance
import org.codeberg.quecomet.oshi.ui.components.Card
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.darken
import org.codeberg.quecomet.oshi.utils.copyToClipboard
import java.util.Date

@Composable
fun HistoryItem(
    entity: UploadedFileAndOshiInstance,
    modifier: Modifier = Modifier,
    onClick: () -> Unit,
    onDelete: (UploadedFile) -> Unit,
) {
  val context = LocalContext.current
  Card(modifier, onClick) {
    Column(
        modifier =
            Modifier.padding(
                vertical = dimensionResource(R.dimen.padding_md),
                horizontal = dimensionResource(R.dimen.padding_lg),
            ),
        verticalArrangement = Arrangement.spacedBy(7.dp),
    ) {
      Row(
          horizontalArrangement = Arrangement.SpaceBetween,
          modifier = Modifier.fillMaxWidth(),
      ) {
        Column {
          Text(
              entity.uploadedFile.originalFilename,
              modifier = Modifier.widthIn(0.dp, 180.dp),
              maxLines = 1,
              overflow = TextOverflow.Ellipsis,
          )
          Text(
              entity.oshiInstance.toString(),
              fontSize = 13.sp,
              color = MaterialTheme.colorScheme.onSurfaceVariant.darken(0.84f),
              modifier = Modifier.widthIn(0.dp, 160.dp),
              maxLines = 1,
              overflow = TextOverflow.Ellipsis,
          )
        }
        Column(horizontalAlignment = Alignment.End) {
          Text(
              text =
                  if (entity.uploadedFile.expiresAt.time < Date().time)
                      stringResource(R.string.expired)
                  else
                      stringResource(
                          R.string.expires_in,
                          DateUtils.getRelativeTimeSpanString(
                                  entity.uploadedFile.expiresAt.time,
                                  System.currentTimeMillis(),
                                  0L,
                                  DateUtils.FORMAT_ABBREV_ALL,
                              )
                              .toString()),
              fontSize = 13.sp,
          )
          Text(
              text =
                  stringResource(
                      R.string.created_at_relative_timestamp,
                      DateUtils.getRelativeTimeSpanString(
                              entity.uploadedFile.createdAt.time,
                              System.currentTimeMillis(),
                              0L,
                              DateUtils.FORMAT_ABBREV_ALL,
                          )
                          .toString()),
              fontSize = 13.sp,
              color = MaterialTheme.colorScheme.onSurfaceVariant.darken(0.84f),
          )
        }
      }

      HorizontalDivider()

      Row(horizontalArrangement = Arrangement.SpaceBetween, modifier = Modifier.fillMaxWidth()) {
        Row(horizontalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_md))) {
          OutlinedButton(
              onClick = {
                copyToClipboard(
                    context,
                    "${entity.oshiInstance.protocol.name}://${entity.oshiInstance.host}${entity.uploadedFile.downloadPath}")
              },
          ) {
            Icon(Icons.Default.ContentCopy, null)
            Spacer(Modifier.width(4.dp))
            Text(stringResource(R.string.link))
          }
          if (!entity.oshiInstance.alternateHost.isNullOrBlank()) {
            OutlinedButton(
                onClick = {
                  copyToClipboard(
                      context,
                      "${entity.oshiInstance.alternateProtocol.name}://${entity.oshiInstance.alternateHost}${entity.uploadedFile.downloadPath}")
                },
            ) {
              Icon(Icons.Default.ContentCopy, null)
              Spacer(Modifier.width(4.dp))
              Text(stringResource(R.string.alternate_link))
            }
          }
        }
        OutlinedIconButton(
            onClick = { onDelete(entity.uploadedFile) },
            border = BorderStroke(1.dp, MaterialTheme.colorScheme.error)) {
              Icon(
                  imageVector = Icons.Default.DeleteForever,
                  contentDescription = stringResource(R.string.delete),
                  tint = MaterialTheme.colorScheme.error,
              )
            }
      }
    }
  }
}

@Preview("default")
@Composable
fun HistoryItemPreview() {
  OshiTheme(darkTheme = false) {
    HistoryItem(
        UploadedFileRepositoryMock.filesAndOshiInstances[0],
        onClick = {},
        onDelete = {},
    )
  }
}

@Preview("default -dark")
@Composable
fun HistoryItemDarkPreview() {
  OshiTheme(darkTheme = true) {
    HistoryItem(
        UploadedFileRepositoryMock.filesAndOshiInstances[0],
        onClick = {},
        onDelete = {},
    )
  }
}
