package org.codeberg.quecomet.oshi.ui.screens.history

import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.exclude
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.navigationBars
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Search
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.ScaffoldDefaults
import androidx.compose.material3.SearchBar
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.paging.LoadState
import androidx.paging.compose.collectAsLazyPagingItems
import androidx.paging.compose.itemKey
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.UploadedFileRepositoryMock
import org.codeberg.quecomet.oshi.data.room.UploadedFile
import org.codeberg.quecomet.oshi.model.SnackbarMessage
import org.codeberg.quecomet.oshi.model.UserMessage
import org.codeberg.quecomet.oshi.ui.components.Card
import org.codeberg.quecomet.oshi.ui.components.ConfirmDeleteDialog
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.verticalFadingEdge
import org.codeberg.quecomet.oshi.utils.isWorking

@OptIn(ExperimentalMaterial3Api::class, ExperimentalFoundationApi::class)
@Composable
fun History(
    viewModel: HistoryViewModel,
    onUploadedFileSelected: (managePath: String, oshiInstanceId: Int) -> Unit,
    modifier: Modifier = Modifier
) {
  val pagingItems = viewModel.uploadedFileAndOshiInstancePagingDataFlow.collectAsLazyPagingItems()
  val listState = rememberLazyListState()
  val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())
  var deletingItem by remember { mutableStateOf<UploadedFile?>(null) }
  val searchTerm by viewModel.searchTerm.collectAsStateWithLifecycle()

  LaunchedEffect(key1 = pagingItems.loadState.refresh) {
    if (pagingItems.loadState.refresh is LoadState.Error) {
      val errorMsg = (pagingItems.loadState.refresh as LoadState.Error).error.localizedMessage
      SnackbarManager.showMessage(
          SnackbarMessage.from(
              message =
                  if (errorMsg.isNullOrEmpty()) UserMessage.from(R.string.unknown_error)
                  else UserMessage.from(errorMsg),
          ),
      )
    }
  }

  Scaffold(
      modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
      snackbarHost = { SnackbarHost(hostState = SnackbarManager.snackbarHostState) },
      topBar = {
        CenterAlignedTopAppBar(
            scrollBehavior = scrollBehavior,
            colors =
                TopAppBarDefaults.centerAlignedTopAppBarColors(
                    scrolledContainerColor = MaterialTheme.colorScheme.inverseOnSurface,
                ),
            title = { Text(stringResource(R.string.history)) },
        )
      },
      contentWindowInsets =
          ScaffoldDefaults.contentWindowInsets.exclude(WindowInsets.navigationBars),
  ) { padding ->
    if (deletingItem != null) {
      ConfirmDeleteDialog(
          itemLabel = deletingItem!!.filename,
          workState = viewModel.deleteWorkState,
          onConfirm = { viewModel.deleteUploadedFile(deletingItem!!) { deletingItem = null } },
          onDismissRequest = {
            if (!viewModel.deleteWorkState.isWorking()) {
              deletingItem = null
            }
          })
    }
    Box(
        modifier =
            modifier
                .padding(padding)
                .padding(horizontal = dimensionResource(R.dimen.page_padding))
                .fillMaxSize(),
    ) {
      when (pagingItems.loadState.refresh) {
        is LoadState.Error ->
            Card {
              Column(Modifier.padding(vertical = 70.dp, horizontal = 20.dp)) {
                Text(stringResource(R.string.unknown_error))
              }
            }
        else -> {
          Column(
              verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.page_padding)),
              horizontalAlignment = Alignment.CenterHorizontally) {
                SearchBar(
                    modifier = Modifier.fillMaxWidth(),
                    query = searchTerm,
                    windowInsets = WindowInsets(0.dp, 0.dp, 0.dp, 0.dp),
                    onSearch = {},
                    onQueryChange = { viewModel.updateSearchTerm(it) },
                    active = false,
                    onActiveChange = {},
                    placeholder = { Text(text = stringResource(R.string.search_placeholder)) },
                    leadingIcon = {
                      Icon(imageVector = Icons.Default.Search, contentDescription = null)
                    },
                ) {}

                if (pagingItems.itemCount == 0) {
                  Column {
                    Card {
                      Column(
                          Modifier.padding(
                              horizontal = dimensionResource(R.dimen.padding_md),
                              vertical = 70.dp,
                          ),
                          horizontalAlignment = Alignment.CenterHorizontally,
                      ) {
                        Text(
                            if (searchTerm.isBlank()) stringResource(R.string.notice_empty_history)
                            else stringResource(R.string.search_had_no_result),
                            fontSize = 22.sp,
                            textAlign = TextAlign.Center,
                        )
                      }
                    }
                  }
                } else {
                  LazyColumn(
                      state = listState,
                      modifier =
                          Modifier.fillMaxSize()
                              .verticalFadingEdge(
                                  listState, dimensionResource(R.dimen.fading_edge)),
                      horizontalAlignment = Alignment.CenterHorizontally,
                      verticalArrangement =
                          Arrangement.spacedBy(dimensionResource(R.dimen.page_padding)),
                  ) {
                    if (pagingItems.loadState.refresh is LoadState.Loading) {
                      item {
                        CircularProgressIndicator(
                            modifier = Modifier.size(50.dp).animateItemPlacement(),
                        )
                      }
                    }
                    items(
                        count = pagingItems.itemCount,
                        key =
                            pagingItems.itemKey {
                              it.uploadedFile.oshiInstanceId.toString() + it.uploadedFile.managePath
                            }) { index ->
                          val uploadedFileAndOshiInstance = pagingItems[index]
                          if (uploadedFileAndOshiInstance != null) {
                            HistoryItem(
                                uploadedFileAndOshiInstance,
                                modifier = Modifier.animateItemPlacement(),
                                onClick = {
                                  onUploadedFileSelected(
                                      uploadedFileAndOshiInstance.uploadedFile.managePath,
                                      uploadedFileAndOshiInstance.uploadedFile.oshiInstanceId,
                                  )
                                },
                                onDelete = { deletingItem = it },
                            )
                          }
                        }
                    item {
                      if (pagingItems.loadState.append == LoadState.Loading) {
                        CircularProgressIndicator(
                            modifier = Modifier.padding(16.dp).animateItemPlacement())
                      }
                    }
                  }
                }
              }
        }
      }
    }
  }
}

@Preview("default")
@Preview("large font", fontScale = 2f)
@Composable
fun HistoryPreview() {
  OshiTheme {
    History(
        HistoryViewModel(SavedStateHandle(), UploadedFileRepositoryMock()),
        onUploadedFileSelected = { _, _ -> },
    )
  }
}
