package org.codeberg.quecomet.oshi.network

import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.engine.okhttp.OkHttp
import io.ktor.client.plugins.HttpRequestRetry
import io.ktor.client.plugins.HttpResponseValidator
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.UserAgent
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.plugins.logging.ANDROID
import io.ktor.client.plugins.logging.LogLevel
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.http.ContentType
import io.ktor.http.HttpMethod
import io.ktor.http.contentType
import io.ktor.http.isSuccess
import io.ktor.serialization.ContentConvertException
import okhttp3.ConnectionSpec.Companion.CLEARTEXT
import okhttp3.ConnectionSpec.Companion.MODERN_TLS
import okhttp3.ConnectionSpec.Companion.RESTRICTED_TLS
import okhttp3.Dns
import okhttp3.internal.tls.OkHostnameVerifier
import org.codeberg.quecomet.oshi.BuildConfig
import org.codeberg.quecomet.oshi.UserSettings
import org.codeberg.quecomet.oshi.data.room.OshiInstance
import org.codeberg.quecomet.oshi.model.api.GeneralResponse
import org.codeberg.quecomet.oshi.model.api.exceptions.ApiException
import java.net.InetAddress
import javax.net.ssl.SSLContext
import javax.net.ssl.X509TrustManager

private val connectionSpecs =
    listOf(
        RESTRICTED_TLS, // order matters here, so we put restricted before modern
        MODERN_TLS,
        CLEARTEXT,
    )

private const val TIMEOUT_MILLIS_HIGH = 50_000L
private const val TIMEOUT_MILLIS_LOW = 15_000L

fun createHttpClient(
    userSettings: UserSettings,
    oshiInstance: OshiInstance,
    /** You don't need to pass this unless you know what you are doing. */
    trustManager: X509TrustManager? = null,
    userAgent: String? = null,
): HttpClient {
  val proxyConfig = getProxyConfig(userSettings)
  var customX509TrustManager = trustManager
  if (customX509TrustManager == null) {
    customX509TrustManager = CustomX509TrustManager(userSettings.sslCheckMode, oshiInstance)
  }

  return HttpClient(OkHttp) {
    expectSuccess = true // throw on non-2xx
    followRedirects = false

    engine {
      proxy = proxyConfig
      config {
        if (proxyConfig.isTor()) {
          dns(NoDns())
        }
        connectionSpecs(connectionSpecs)
        hostnameVerifier { hostname, session ->
          session?.sessionContext?.sessionTimeout = 60

          if (hostname.endsWith(".onion")) {
            return@hostnameVerifier hostname == oshiInstance.host
          }

          // use default hostname verifier otherwise
          return@hostnameVerifier OkHostnameVerifier.verify(hostname, session)
        }

        // setup custom x509 trust manager
        val sslContext = SSLContext.getInstance("TLS")
        sslContext.init(null, arrayOf(customX509TrustManager), null)
        sslSocketFactory(sslContext.socketFactory, customX509TrustManager)
      }
    }

    install(HttpTimeout) {
      if (proxyConfig.isTor()) {
        requestTimeoutMillis = TIMEOUT_MILLIS_HIGH
        connectTimeoutMillis = TIMEOUT_MILLIS_HIGH
        socketTimeoutMillis = TIMEOUT_MILLIS_HIGH
      } else {
        requestTimeoutMillis = TIMEOUT_MILLIS_LOW
        connectTimeoutMillis = TIMEOUT_MILLIS_LOW
        socketTimeoutMillis = TIMEOUT_MILLIS_LOW
      }
    }

    install(UserAgent) {
      agent = userAgent ?: "org.codeberg.quecomet.oshi uploader android v${BuildConfig.VERSION_NAME}"
    }

    defaultRequest {
      url {
        protocol = oshiInstance.protocol
        host = oshiInstance.host
      }
    }

    install(HttpRequestRetry) {
      maxRetries = 2
      retryIf { request, response ->
        request.method == HttpMethod.Get && !response.status.isSuccess()
      }
      exponentialDelay()
    }

    HttpResponseValidator {
      validateResponse { response ->
        val contentType = response.contentType()
        if (contentType?.match(ContentType.Application.Json) == true) {
          try {
            val resp: GeneralResponse = response.body()
            if (!resp.success) {
              if (resp.error != null) {
                throw ApiException(resp.error)
              } else {
                throw ApiException("Unknown error.")
              }
            }
          } catch (e: ContentConvertException) {
            // no need to handle
          }
        }
      }
    }

    if (BuildConfig.DEBUG) {
      install(Logging) {
        logger = Logger.ANDROID
        level = LogLevel.HEADERS
      }
    }
  }
}

/** Prevent DNS requests. Important when proxying all requests over Tor to not leak DNS queries. */
private class NoDns : Dns {
  override fun lookup(hostname: String): List<InetAddress> {
    return listOf(InetAddress.getByAddress(hostname, ByteArray(4)))
  }
}
