package org.codeberg.quecomet.oshi.model.api

import androidx.compose.runtime.Immutable
import okhttp3.internal.toLongOrDefault
import java.time.LocalDateTime
import java.time.ZoneOffset
import java.time.format.DateTimeFormatter
import java.time.format.DateTimeFormatterBuilder
import java.util.Date

@Immutable
data class ManageFileCurlResponse(
    val filename: String,
    /** clear net url */
    val mainDownload: String,
    /** tor url */
    val torDownload: String,
    val size: Long,
    val type: String,
    val hashsumSha1: String,
    val created: Date,
    val expires: Date,
    val destroyAfterDownload: Boolean,
    val onionOnly: Boolean,
    val hits: String,
    /** url for deleting the file */
    val deleteFile: String,
)

fun parseManageFileCurlResponse(response: String): ManageFileCurlResponse {
  val map = parseCurlResponse(response)
  return ManageFileCurlResponse(
      filename = map.getOrDefault("filename", ""),
      mainDownload = map.getOrDefault("main_download", ""),
      torDownload = map.getOrDefault("tor_download", ""),
      size = map.getOrDefault("size", "0").toLongOrDefault(0),
      type = map.getOrDefault("type", ""),
      hashsumSha1 = map.getOrDefault("hashsum", "").split(' ')[0],
      created =
          parseOshiDate(
              map.getOrDefault("created", "Thu Jan 01 03:30:00 1970"),
          ) ?: error("Invalid date format"),
      expires =
          parseOshiDate(
              map.getOrDefault("expires", "Thu Jan 01 03:30:00 1970"),
          ) ?: error("Invalid date format"),
      destroyAfterDownload =
          map.getOrDefault("destroy_after_download", "No").lowercase().let { it != "no" },
      onionOnly = map.getOrDefault("onion_only", "No").lowercase().let { it != "no" },
      hits = map.getOrDefault("hits", "0/0"),
      deleteFile = map.getOrDefault("delete_file", ""),
  )
}

fun parseOshiDate(date: String?): Date? {
  val dateTimeFormatterBuilder =
      DateTimeFormatterBuilder().append(DateTimeFormatter.ofPattern("EEE MMM dd HH:mm:ss yyyy"))
  val dateTimeFormatter = dateTimeFormatterBuilder.toFormatter()
  val localDateTime = LocalDateTime.parse(date, dateTimeFormatter)
  return Date.from(localDateTime.toInstant(ZoneOffset.UTC))
}
