package org.codeberg.quecomet.oshi.data.room

import androidx.paging.PagingSource
import androidx.room.Dao
import androidx.room.Insert
import androidx.room.OnConflictStrategy
import androidx.room.Query
import androidx.room.Transaction
import kotlinx.coroutines.flow.Flow

/** [Room] DAO for [OshiInstance] related operations. */
@Dao
abstract class OshiInstanceDao : BaseDao<OshiInstance> {

  @Query("SELECT * FROM `OshiInstance` WHERE id = :id")
  abstract suspend fun getInstanceWithId(id: Int): OshiInstance?

  @Query("SELECT * FROM `OshiInstance` WHERE id = :id")
  abstract fun getInstanceWithIdFlow(id: Int): Flow<OshiInstance?>

  @Query("SELECT * FROM OshiInstance ORDER BY createdAt ASC LIMIT 1")
  abstract suspend fun getFirstInstance(): OshiInstance?

  @Transaction
  @Query("SELECT * FROM OshiInstance ORDER BY createdAt ASC")
  abstract fun getInstancesSortedByCreatedAtPagingSource(): PagingSource<Int, OshiInstance>

  @Transaction
  @Query(
      "SELECT * FROM OshiInstance WHERE label LIKE '%' || :searchTerm || '%' or host LIKE '%' || :searchTerm || '%' ORDER BY createdAt ASC")
  abstract fun getInstancesSortedByCreatedAtPagingSourceFiltered(
      searchTerm: String
  ): PagingSource<Int, OshiInstance>

  @Query("DELETE FROM `OshiInstance` WHERE id = :id") abstract suspend fun delete(id: Int)

  @Insert(onConflict = OnConflictStrategy.ABORT)
  abstract override suspend fun insert(entity: OshiInstance): Long

  @Transaction
  @Query(
      "SELECT OshiInstance.id FROM UploadWork join OshiInstance WHERE OshiInstance.id NOT IN (:ids) GROUP by oshiInstanceId ORDER BY UploadWork.createdAt DESC LIMIT 1")
  abstract suspend fun getLatestUsedOshiInstanceIdExcept(ids: List<Int>): Int?
}
