package org.codeberg.quecomet.oshi.data

import androidx.compose.runtime.Stable
import androidx.paging.Pager
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import dagger.Binds
import dagger.Module
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import io.ktor.client.plugins.ClientRequestException
import io.ktor.client.plugins.expectSuccess
import io.ktor.client.request.delete
import io.ktor.client.request.get
import io.ktor.client.statement.bodyAsText
import io.ktor.http.HttpStatusCode
import kotlinx.coroutines.flow.Flow
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.room.OshiDatabase
import org.codeberg.quecomet.oshi.data.room.UploadedFile
import org.codeberg.quecomet.oshi.data.room.UploadedFileAndOshiInstance
import org.codeberg.quecomet.oshi.data.room.UploadedFileWithDeletedInRemote
import org.codeberg.quecomet.oshi.data.room.UploadedFileWithDestroyAfterDl
import org.codeberg.quecomet.oshi.data.room.UploadedFileWithIsOnionOnly
import org.codeberg.quecomet.oshi.exceptions.LocalizedException
import org.codeberg.quecomet.oshi.model.api.exceptions.FailedToDeleteFileException
import org.codeberg.quecomet.oshi.model.api.parseManageFileCurlResponse
import org.codeberg.quecomet.oshi.network.createHttpClient
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
@Stable
class UploadedFileRepositoryImpl
@Inject
constructor(
    private val oshiDatabase: OshiDatabase,
    private val userSettingsRepository: UserSettingsRepository,
    private val oshiInstanceRepository: OshiInstanceRepository,
) : UploadedFileRepository {

  override suspend fun insertUploadedFile(uploadedFile: UploadedFile) {
    oshiDatabase.uploadedFilesDao().insert(uploadedFile)
  }

  override suspend fun getUploadedFileAndOshiInstance(
      managePath: String,
      oshiInstanceId: Int
  ): UploadedFileAndOshiInstance? {
    return oshiDatabase
        .uploadedFilesDao()
        .uploadedFileAndOshiInstanceByManagePath(managePath, oshiInstanceId)
  }

  override fun getUploadedFileAndOshiInstanceFlow(
      managePath: String,
      oshiInstanceId: Int
  ): Flow<UploadedFileAndOshiInstance?> {
    return oshiDatabase
        .uploadedFilesDao()
        .uploadedFileAndOshiInstanceByManagePathFlow(managePath, oshiInstanceId)
  }

  override fun getUploadedFileAndOshiInstanceList(
      searchTerm: String,
      pageSize: Int
  ): Flow<PagingData<UploadedFileAndOshiInstance>> {
    return Pager(
            config = PagingConfig(pageSize = pageSize),
            pagingSourceFactory = {
              if (searchTerm.isBlank()) {
                oshiDatabase
                    .uploadedFilesDao()
                    .uploadedFileAndOshiInstanceSortedByCreatedAtPagingSource()
              } else {
                oshiDatabase
                    .uploadedFilesDao()
                    .uploadedFileAndOshiInstanceSortedByCreatedAtPagingSourceFiltered(searchTerm)
              }
            },
        )
        .flow
  }

  override suspend fun count(): Int {
    return oshiDatabase.uploadedFilesDao().count()
  }

  override suspend fun deleteRemote(managePath: String, oshiInstanceId: Int) {
    val entity =
        this.getUploadedFileAndOshiInstance(managePath, oshiInstanceId)
            ?: throw RuntimeException("file with id $managePath not found")
    val userSettings = userSettingsRepository.fetchSettings()
    val resp = createHttpClient(userSettings, entity.oshiInstance).delete(managePath)

    if (resp.status != HttpStatusCode.OK && resp.status != HttpStatusCode.NotFound) {
      throw FailedToDeleteFileException()
    }

    oshiDatabase
        .uploadedFilesDao()
        .setDeletedInRemote(UploadedFileWithDeletedInRemote(managePath, oshiInstanceId, true))
  }

  override suspend fun deleteLocal(managePath: String, oshiInstanceId: Int) {
    oshiDatabase.uploadedFilesDao().delete(managePath, oshiInstanceId)
  }

  override suspend fun delete(managePath: String, oshiInstanceId: Int) {
      try {
        deleteRemote(managePath, oshiInstanceId)
      } catch (e: ClientRequestException) {
        if (e.response.status != HttpStatusCode.NotFound) {
          throw e
        }
      }
      deleteLocal(managePath, oshiInstanceId)
  }

  override suspend fun fetchUploadedFile(managePath: String, oshiInstanceId: Int) {
    val entity =
        oshiDatabase
            .uploadedFilesDao()
            .uploadedFileAndOshiInstanceByManagePath(managePath, oshiInstanceId)
            ?: throw LocalizedException(R.string.exception_uploaded_file_not_found_locally)

    val httpClient =
        createHttpClient(
            userSettingsRepository.fetchSettings(),
            entity.oshiInstance,
            userAgent = "curl/8.4.0",
        )

    val curlResp =
        httpClient
            .get(entity.uploadedFile.managePath) {
              url {
                protocol = entity.oshiInstance.protocol
                host = entity.oshiInstance.host
              }
            }
            .bodyAsText()

    val parsedCurlResponse = parseManageFileCurlResponse(curlResp)

    oshiDatabase
        .uploadedFilesDao()
        .upsert(
            entity.uploadedFile.copy(
                hits = parsedCurlResponse.hits,
                filename = parsedCurlResponse.filename,
                isOnionOnly = parsedCurlResponse.onionOnly,
                hashsumSha1 = parsedCurlResponse.hashsumSha1,
                destroyAfterDl = parsedCurlResponse.destroyAfterDownload,
                createdAt = parsedCurlResponse.created,
                expiresAt = parsedCurlResponse.expires,
                size = parsedCurlResponse.size,
                mimeType = parsedCurlResponse.type,
            ))
  }

  override suspend fun toggleDestroyAfterDl(managePath: String, oshiInstanceId: Int) {
    val entity =
        oshiDatabase
            .uploadedFilesDao()
            .uploadedFileAndOshiInstanceByManagePath(managePath, oshiInstanceId)
            ?: throw LocalizedException(R.string.exception_uploaded_file_not_found_locally)

    val httpClient =
        createHttpClient(
            userSettingsRepository.fetchSettings(),
            entity.oshiInstance,
        )

    val resp =
        httpClient.get(entity.uploadedFile.managePath) {
          url {
            protocol = entity.oshiInstance.protocol
            host = entity.oshiInstance.host
            parameters.append("toggleautodestroy", "1")
          }
          setAttributes { expectSuccess = false }
        }

    if (resp.status.value != 302) {
      throw LocalizedException(R.string.exception_failed_to_toggle_destroy_after_dl)
    }

    oshiDatabase
        .uploadedFilesDao()
        .setDestroyAfterDl(
            UploadedFileWithDestroyAfterDl(
                managePath = managePath,
                oshiInstanceId = oshiInstanceId,
                destroyAfterDl = !entity.uploadedFile.destroyAfterDl,
            ))
  }

  override suspend fun toggleOnionOnly(managePath: String, oshiInstanceId: Int) {
    val entity =
        oshiDatabase
            .uploadedFilesDao()
            .uploadedFileAndOshiInstanceByManagePath(managePath, oshiInstanceId)
            ?: throw LocalizedException(R.string.exception_uploaded_file_not_found_locally)

    val httpClient =
        createHttpClient(
            userSettingsRepository.fetchSettings(),
            entity.oshiInstance,
        )

    val resp =
        httpClient.get(entity.uploadedFile.managePath) {
          url {
            protocol = entity.oshiInstance.protocol
            host = entity.oshiInstance.host
            parameters.append("toggleoniononly", "1")
          }
          setAttributes { expectSuccess = false }
        }

    if (resp.status.value != 302) {
      throw LocalizedException(R.string.exception_failed_to_toggle_onion_only)
    }

    oshiDatabase
        .uploadedFilesDao()
        .setIsOnionOnly(
            UploadedFileWithIsOnionOnly(
                managePath = managePath,
                oshiInstanceId = oshiInstanceId,
                isOnionOnly = !entity.uploadedFile.isOnionOnly,
            ))
  }
}

@Module
@InstallIn(SingletonComponent::class)
abstract class FileRepositoryModule {
  @Binds
  @Singleton
  abstract fun provideFileRepositoryModule(
      fileRepositoryImpl: UploadedFileRepositoryImpl
  ): UploadedFileRepository
}
