package org.codeberg.quecomet.oshi.data

import androidx.compose.runtime.Stable
import androidx.paging.PagingData
import io.ktor.http.URLProtocol
import kotlinx.collections.immutable.persistentListOf
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import org.codeberg.quecomet.oshi.data.room.OshiInstance

@Stable
class OshiInstanceRepositoryMock : OshiInstanceRepository {

  companion object {

    val instances =
        persistentListOf(
            OshiInstance(
                id = 1,
                host = "oshi.ec",
                protocol = URLProtocol.HTTPS,
                label = "oshi.ec",
            ),
            OshiInstance(
                id = 2,
                host = "5ety7tpkim5me6eszuwcje7bmy25pbtrjtue7zkqqgziljwqy3rrikqd.onion",
                protocol = URLProtocol.HTTPS,
                label = "oshi.ec",
            ),
            OshiInstance(
                id = 3,
                host = "oshi.ec",
                protocol = URLProtocol.HTTPS,
                label = "oshi.ec",
                sha1Fingerprint =
                    "a1:b2:c3:d4:a1:b2:c3:d4:a1:b2:c3:d4:a1:b2:c3:d4:a1:b2:c3:d4:a1:b2",
                sha256Fingerprint =
                    "a1:b2:c3:d4:a1:b2:c3:d4:a1:b2:c3:d4:a1:b2:c3:d4:a1:b2:c3:d4:a1:b2"),
        )
  }

  override fun getInstanceFlow(id: Int): Flow<OshiInstance?> {
    return flow { emit(instances[id]) }
  }

  override fun getInstanceList(searchTerm: String, pageSize: Int): Flow<PagingData<OshiInstance>> {
    return flow {
      emit(
          PagingData.from(instances),
      )
    }
  }

  fun getEmptyInstanceList(): Flow<PagingData<OshiInstance>> {
    return flow {
      emit(
          PagingData.from(emptyList()),
      )
    }
  }

  override suspend fun getInstance(id: Int): OshiInstance {
    return instances[id]
  }

  override suspend fun getFirstInstance(): OshiInstance {
    return instances[0]
  }

  override suspend fun addInstance(instance: OshiInstance) {}

  override suspend fun delete(id: Int) {}

  override suspend fun updateInstance(instance: OshiInstance) {}
}
