package org.codeberg.quecomet.oshi.data

import androidx.compose.runtime.Stable
import androidx.paging.Pager
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import dagger.Binds
import dagger.Module
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import javax.inject.Inject
import javax.inject.Singleton
import kotlinx.coroutines.flow.Flow
import org.codeberg.quecomet.oshi.data.room.OshiDatabase
import org.codeberg.quecomet.oshi.data.room.OshiInstance

@Singleton
@Stable
class OshiInstanceRepositoryImpl
@Inject
constructor(
    private val oshiDatabase: OshiDatabase,
    private val userSettingsRepository: UserSettingsRepository,
) : OshiInstanceRepository {

  override fun getInstanceFlow(id: Int): Flow<OshiInstance?> {
    return oshiDatabase.oshiInstancesDao().getInstanceWithIdFlow(id)
  }

  override fun getInstanceList(searchTerm: String, pageSize: Int): Flow<PagingData<OshiInstance>> {
    return Pager(
            config = PagingConfig(pageSize = pageSize),
            pagingSourceFactory = {
              if (searchTerm.isBlank()) {
                oshiDatabase.oshiInstancesDao().getInstancesSortedByCreatedAtPagingSource()
              } else {
                oshiDatabase
                    .oshiInstancesDao()
                    .getInstancesSortedByCreatedAtPagingSourceFiltered(searchTerm)
              }
            },
        )
        .flow
  }

  override suspend fun getInstance(id: Int): OshiInstance? {
    return oshiDatabase.oshiInstancesDao().getInstanceWithId(id)
  }

  override suspend fun getFirstInstance(): OshiInstance? {
    return oshiDatabase.oshiInstancesDao().getFirstInstance()
  }

  override suspend fun addInstance(instance: OshiInstance) {
    oshiDatabase.oshiInstancesDao().insert(instance)
  }

  override suspend fun delete(id: Int) {
    oshiDatabase.oshiInstancesDao().delete(id)
  }

  override suspend fun updateInstance(instance: OshiInstance) {
    oshiDatabase.oshiInstancesDao().update(instance)
  }
}

@Module
@InstallIn(SingletonComponent::class)
abstract class OshiInstanceRepositoryModule {
  @Binds
  @Singleton
  abstract fun provideOshiInstanceRepositoryModule(
      oshiInstanceRepositoryImpl: OshiInstanceRepositoryImpl
  ): OshiInstanceRepository
}
