package org.codeberg.quecomet.oshi

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import androidx.compose.runtime.Stable
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.update
import org.codeberg.quecomet.oshi.ui.navigation.getManageFileUri
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
@Stable
class DeepLinkHandler @Inject constructor() {
  val deepLinkEvent = MutableStateFlow<DeepLinkEvent>(DeepLinkEvent.None)

  fun handleDeepLink(intent: Intent?) {
    if (intent != null) {
      // make a copy
      val newIntent = Intent(intent)
      // this is to remove the FLAG_ACTIVITY_NEW_TASK flag, because navigation bugs otherwise
      newIntent.flags = Intent.FLAG_ACTIVITY_SINGLE_TOP and Intent.FLAG_ACTIVITY_CLEAR_TOP

      deepLinkEvent.update { DeepLinkEvent.NavigateWithDeepLink(newIntent) }
    }
  }

  fun consumeEvent() {
    deepLinkEvent.update { DeepLinkEvent.None }
  }
}

sealed interface DeepLinkEvent {

  @Stable data class NavigateWithDeepLink(val intent: Intent) : DeepLinkEvent

  object None : DeepLinkEvent
}

fun getManageFileDeepLinkPendingIntent(
    context: Context,
    managePath: String,
    oshiInstanceId: Int
): PendingIntent {
  val routeIntent =
      Intent(Intent.ACTION_VIEW, getManageFileUri(managePath, oshiInstanceId)).apply {
        flags = Intent.FLAG_ACTIVITY_SINGLE_TOP and Intent.FLAG_ACTIVITY_CLEAR_TOP
      }

  val flags = PendingIntent.FLAG_IMMUTABLE or PendingIntent.FLAG_UPDATE_CURRENT

  return PendingIntent.getActivity(context, 0, routeIntent, flags)
}
