package org.codeberg.quecomet.oshi.worker

import android.annotation.SuppressLint
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.util.Log
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import androidx.core.app.PendingIntentCompat
import androidx.core.net.toUri
import androidx.hilt.work.HiltWorker
import androidx.work.CoroutineWorker
import androidx.work.Data
import androidx.work.WorkerParameters
import androidx.work.workDataOf
import dagger.assisted.Assisted
import dagger.assisted.AssistedInject
import io.ktor.client.plugins.HttpTimeoutConfig.Companion.INFINITE_TIMEOUT_MS
import io.ktor.client.plugins.expectSuccess
import io.ktor.client.plugins.onUpload
import io.ktor.client.plugins.timeout
import io.ktor.client.request.forms.InputProvider
import io.ktor.client.request.forms.formData
import io.ktor.client.request.forms.submitFormWithBinaryData
import io.ktor.client.statement.bodyAsText
import io.ktor.http.Headers
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpMethod
import io.ktor.http.quote
import io.ktor.utils.io.streams.asInput
import org.codeberg.quecomet.oshi.ACTION_CANCEL_UPLOAD
import org.codeberg.quecomet.oshi.ACTION_RETRY_UPLOAD
import org.codeberg.quecomet.oshi.ACTION_RETRY_UPLOAD_DIFFERENT_INSTANCE
import org.codeberg.quecomet.oshi.MainActivity
import org.codeberg.quecomet.oshi.MyNotificationActionReceiver
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.OshiInstanceRepository
import org.codeberg.quecomet.oshi.data.UploadWorkRepository
import org.codeberg.quecomet.oshi.data.UploadedFileRepository
import org.codeberg.quecomet.oshi.data.UserSettingsRepository
import org.codeberg.quecomet.oshi.data.room.UploadWorkAndOshiInstance
import org.codeberg.quecomet.oshi.data.room.UploadedFile
import org.codeberg.quecomet.oshi.exceptions.ContentTooLargeUploadException
import org.codeberg.quecomet.oshi.exceptions.UnrecoverableUploadException
import org.codeberg.quecomet.oshi.getManageFileDeepLinkPendingIntent
import org.codeberg.quecomet.oshi.model.api.parseUploadFileCurlResponse
import org.codeberg.quecomet.oshi.network.createHttpClient
import org.codeberg.quecomet.oshi.utils.createNotificationChannels
import org.codeberg.quecomet.oshi.utils.getFileSize
import org.codeberg.quecomet.oshi.utils.getUploadWorkNotifTag
import org.codeberg.quecomet.oshi.utils.isNotificationAllowed
import java.io.FileNotFoundException
import java.util.Date
import java.util.concurrent.CancellationException

@HiltWorker
class UploadFileWorker
@AssistedInject
constructor(
    @Assisted context: Context,
    @Assisted workerParams: WorkerParameters,
    private val uploadWorkRepository: UploadWorkRepository,
    private val uploadedFileRepository: UploadedFileRepository,
    private val userSettingsRepository: UserSettingsRepository,
    private val oshiInstanceRepository: OshiInstanceRepository,
) : CoroutineWorker(context, workerParams) {

  companion object {
    const val TAG: String = "UploadFileWorker"
  }

  private fun getWorkUri(workUUID: String): Uri {
    return "UploadWork://$workUUID".toUri()
  }

  private fun notifyProgress(
      context: Context,
      notifManager: NotificationManagerCompat?,
      entity: UploadWorkAndOshiInstance,
      progress: Float,
      cancelPendingIntent: PendingIntent,
      activityPendingIntent: PendingIntent,
  ) {
    if (notifManager != null) {

      val notifBuilder =
          NotificationCompat.Builder(
                  context,
                  context.getString(R.string.channel_id_upload_progress),
              )
              .setSmallIcon(R.drawable.ic_notification)
              .setContentTitle(context.getString(R.string.uploading_etc))
              .setContentText(
                  context.getString(
                      R.string.uploading_filename,
                      entity.uploadWork.filename,
                      progress * 100,
                  ),
              )
              .setSubText(
                  context.getString(
                      R.string.uploading_to,
                      entity.oshiInstance.toString(),
                  ),
              )
              .setOngoing(true)
              .setAutoCancel(false)
              .setAllowSystemGeneratedContextualActions(false)
              .setContentIntent(activityPendingIntent)
              .addAction(
                  R.drawable.baseline_cancel_24,
                  context.getString(R.string.cancel),
                  cancelPendingIntent,
              )

      if (progress.compareTo(0f) == 0) {
        notifBuilder.setProgress(100, 0, true)
      } else {
        notifBuilder.setProgress(100, (progress * 100).toInt(), false)
      }

      try {
        notifManager.notify(
            getUploadWorkNotifTag(entity.uploadWork.workUUID),
            R.id.notif_id_upload_progress,
            notifBuilder.build(),
        )
      } catch (e: SecurityException) {
        // ignore
      }
    }
  }

  @SuppressLint("MissingPermission")
  override suspend fun doWork(): Result {
    val context = this.applicationContext

    val notificationAllowed = isNotificationAllowed(context)
    val notifManager: NotificationManagerCompat?
    if (notificationAllowed) {
      notifManager = NotificationManagerCompat.from(context)
      createNotificationChannels(context) // just in case where channel was not created
    } else {
      notifManager = null
    }

    val workUUID =
        this.tags.last { !it.contains("UploadFileWorker") }
            ?: throw UnrecoverableUploadException(
                R.string.exception_worker_data_missing, "could not retrieve workUUID from tags")
    val notifTag = getUploadWorkNotifTag(workUUID)
    var entity: UploadWorkAndOshiInstance? = null

    val activityPendingIntent =
        PendingIntentCompat.getActivity(
            context,
            0,
            Intent(context, MainActivity::class.java).setFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP),
            0,
            false,
        )!!
    var outputData = Data.Builder().build()

    try {
      entity = uploadWorkRepository.getUploadWorkAndOshiInstance(workUUID)
      if (entity == null) {
        notifManager?.cancel(
            getUploadWorkNotifTag(workUUID),
            R.id.notif_id_upload_progress,
        )
        throw UnrecoverableUploadException(
            R.string.exception_work_not_found,
            "UploadWork with workUUID $workUUID not found",
        )
      }

      val cancelIntent =
          Intent(context, MyNotificationActionReceiver::class.java).apply {
            action = ACTION_CANCEL_UPLOAD
            data = getWorkUri(workUUID)
            putExtra("workUUID", workUUID)
            setPackage(context.packageName)
          }
      val cancelPendingIntent: PendingIntent =
          PendingIntentCompat.getBroadcast(context, 0, cancelIntent, 0, false)!!
      notifyProgress(
          context = context,
          notifManager = notifManager,
          entity = entity,
          progress = 0f,
          cancelPendingIntent = cancelPendingIntent,
          activityPendingIntent = activityPendingIntent,
      )

      val isTempFile = entity.uploadWork.fileUri.contains("tempfile")

      val fileUri =
          if (isTempFile) {
            entity.uploadWork.fileUri
                .substring(0, entity.uploadWork.fileUri.length - "?tempfile".length)
                .toUri()
          } else {
            entity.uploadWork.fileUri.toUri()
          }

      val fileSize = getFileSize(fileUri, context)
      var progress = 0f

      setProgress(workDataOf("progress" to progress))

      try {
        context.contentResolver.openInputStream(fileUri).use { inputStream ->
          if (inputStream == null) {
            throw RuntimeException("Cannot open input stream at this moment")
          }

          val httpClient =
              createHttpClient(userSettingsRepository.fetchSettings(), entity.oshiInstance)

          val resp =
              httpClient.submitFormWithBinaryData(
                  formData {
                    append(
                        "f".quote(),
                        InputProvider(fileSize) { inputStream.asInput() },
                        Headers.build {
                          append(
                              HttpHeaders.ContentDisposition,
                              "filename=${entity.uploadWork.filename.quote()}",
                          )
                        },
                    )
                  },
              ) {
                url {
                  protocol = entity.oshiInstance.protocol
                  host = entity.oshiInstance.host
                  method = HttpMethod.Post

                  parameters.append("expire", entity.uploadWork.expireInMinutes.toString())
                  if (entity.uploadWork.destroyAfterDl) {
                    parameters.append("autodestroy", "1")
                  }
                  if (entity.uploadWork.randomizeFilename) {
                    parameters.append("randomizefn", "1")
                  }
                  if (entity.uploadWork.shortenUrl) {
                    parameters.append("shorturl", "1")
                  }
                }
                timeout {
                  // we disable these timeout for uploads, since they may take a long time
                  // depending on size
                  requestTimeoutMillis = INFINITE_TIMEOUT_MS
                  socketTimeoutMillis = INFINITE_TIMEOUT_MS
                  expectSuccess = false
                }
                onUpload { bytesSentTotal, _ ->
                  val newProgress = (bytesSentTotal.toFloat() / fileSize).coerceIn(0f, 1f)
                  if (newProgress - progress >= 0.1 || progress.compareTo(0f) == 0) {
                    progress = newProgress
                    setProgress(workDataOf("progress" to progress))

                    notifyProgress(
                        notifManager = notifManager,
                        context = context,
                        entity = entity,
                        progress = progress,
                        cancelPendingIntent = cancelPendingIntent,
                        activityPendingIntent = activityPendingIntent,
                    )
                  }
                }
              }

          if (resp.status.value == 413) {
            throw ContentTooLargeUploadException(
                R.string.exception_failed_to_upload_content_too_large,
                "413 Content Too Large",
            )
          }

          val respBody = resp.bodyAsText()

          val parsedCurlResponse = parseUploadFileCurlResponse(respBody)

          val uploadedFile =
              UploadedFile(
                  managePath =
                      parsedCurlResponse.manageUrl.toUri().path
                          ?: throw UnrecoverableUploadException(
                              R.string.exception_failed_to_get_manage_url_from_response,
                              "Could not extract manage url from upload response",
                          ),
                  oshiInstanceId = entity.oshiInstance.id,
                  downloadPath =
                      parsedCurlResponse.downloadUrl.toUri().path
                          ?: throw UnrecoverableUploadException(
                              R.string.exception_failed_to_get_download_url_from_response,
                              "Could not extract manage url from upload response",
                          ),
                  filename =
                      "", // this will be changed later when refreshing from the oshi instance
                  originalFilename = entity.uploadWork.filename,
                  size = fileSize,
                  // instance if randomized
                  destroyAfterDl = entity.uploadWork.destroyAfterDl,
                  expiresAt = Date(Date().time + (entity.uploadWork.expireInMinutes.toLong() * 60 * 1000)),
              )

          uploadedFileRepository.insertUploadedFile(uploadedFile)
          uploadWorkRepository.delete(workUUID)
          if (notifManager != null) {

            val notifBuilder =
                NotificationCompat.Builder(
                        context,
                        context.getString(R.string.channel_id_upload_progress),
                    )
                    .setSmallIcon(R.drawable.ic_notification)
                    .setContentTitle(
                        context.getString(R.string.upload_completed),
                    )
                    .setContentText(
                        context.getString(
                            R.string.uploading_filename_succeeded,
                            entity.uploadWork.filename,
                        ),
                    )
                    .setSubText(
                        context.getString(
                            R.string.uploading_to,
                            entity.oshiInstance.toString(),
                        ),
                    )
                    .setAutoCancel(true)
                    .setAllowSystemGeneratedContextualActions(false)
                    .setContentIntent(
                        getManageFileDeepLinkPendingIntent(
                            context, uploadedFile.managePath, entity.oshiInstance.id),
                    )

            notifManager.notify(
                notifTag,
                R.id.notif_id_upload_progress,
                notifBuilder.build(),
            )
          }
          try {
            uploadedFileRepository.fetchUploadedFile(
                uploadedFile.managePath, uploadedFile.oshiInstanceId)
          } catch (e: Throwable) {
            // ignore
          }
        }
        if (isTempFile) {
          try {
            context.contentResolver.delete(fileUri, null, null)
          } catch (e: Throwable) {
            Log.d(TAG, e.message ?: e.toString())
          }
        }
      } catch (e: FileNotFoundException) {
        throw UnrecoverableUploadException(
            R.string.exception_failed_to_open_file,
            e.message ?: e.toString(),
        )
      }
    } catch (e: Throwable) {
      Log.e(TAG, e.message ?: e.toString())
      val isUnrecoverable =
          e is UnrecoverableUploadException ||
              (e is SecurityException &&
                  (e.message ?: e.toString()).lowercase().contains("permission"))
      val isCancelled = e is CancellationException
      if (isUnrecoverable) {
        uploadWorkRepository.setCanRetry(workUUID, false)
        if (e is UnrecoverableUploadException) {
          outputData =
              Data.Builder()
                  .putString(
                      "exception",
                      (e.translation?.let { context.getString(it) } ?: e.message ?: e.toString()))
                  .build()
        }
      }

      if (notifManager != null && entity != null) {
        val notifBuilder =
            NotificationCompat.Builder(
                    context,
                    context.getString(R.string.channel_id_upload_progress),
                )
                .setSmallIcon(R.drawable.ic_notification)
                .setContentTitle(
                    context.getString(
                        if (isCancelled) R.string.upload_cancelled else R.string.upload_failed,
                    ),
                )
                .setContentText(
                    context.getString(
                        if (isCancelled) R.string.uploading_filename_cancelled
                        else R.string.uploading_filename_failed,
                        entity.uploadWork.filename,
                    ),
                )
                .setSubText(
                    context.getString(
                        R.string.uploading_to,
                        entity.oshiInstance.toString(),
                    ),
                )
                .setOngoing(false)
                .setAutoCancel(true)
                .setAllowSystemGeneratedContextualActions(false)
                .setContentIntent(
                    activityPendingIntent,
                )

        if (!isUnrecoverable) {

          notifBuilder.addAction(
              R.drawable.baseline_refresh_24,
              context.getString(R.string.retry),
              PendingIntentCompat.getBroadcast(
                  context,
                  0,
                  Intent(context, MyNotificationActionReceiver::class.java).apply {
                    action = ACTION_RETRY_UPLOAD
                    data = getWorkUri(workUUID)
                    putExtra("workUUID", workUUID)
                    setPackage(context.packageName)
                  },
                  0,
                  false)!!,
          )
          notifBuilder.addAction(
              R.drawable.baseline_refresh_24,
              context.getString(R.string.retry_with_different_instance),
              PendingIntentCompat.getBroadcast(
                  context,
                  0,
                  Intent(context, MyNotificationActionReceiver::class.java).apply {
                    action = ACTION_RETRY_UPLOAD_DIFFERENT_INSTANCE
                    data = getWorkUri(workUUID)
                    putExtra("workUUID", workUUID)
                    setPackage(context.packageName)
                  },
                  0,
                  false)!!,
          )
        }

        notifManager.notify(
            notifTag,
            R.id.notif_id_upload_progress,
            notifBuilder.build(),
        )
      }

      return Result.failure(outputData)
    }

    return Result.success()
  }
}
