package org.codeberg.quecomet.oshi.ui.utils

import androidx.compose.foundation.ScrollState
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.composed
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.draw.drawWithContent
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.debugInspectorInfo
import androidx.compose.ui.unit.Dp

fun Modifier.noRippleClickable(onClick: () -> Unit): Modifier = composed {
  clickable(indication = null, interactionSource = remember { MutableInteractionSource() }) {
    onClick()
  }
}

fun Modifier.horizontalFadingEdge(
  scrollState: ScrollState,
  length: Dp,
  edgeColor: Color? = null,
) =
    composed(
        debugInspectorInfo {
          name = "length"
          value = length
        },
    ) {
      val color = edgeColor ?: Color.Black.copy(alpha = 0.1f)

      drawWithContent {
        val lengthValue = length.toPx()
        val scrollFromStart by derivedStateOf { scrollState.value }
        val scrollFromEnd by derivedStateOf { scrollState.maxValue - scrollState.value }

        val startFadingEdgeStrength =
            lengthValue * (scrollFromStart / lengthValue).coerceAtMost(1f)

        val endFadingEdgeStrength = lengthValue * (scrollFromEnd / lengthValue).coerceAtMost(1f)

        drawContent()

        drawRect(
            brush =
            Brush.horizontalGradient(
                colors =
                listOf(
                    color,
                    Color.Transparent,
                ),
                startX = 0f,
                endX = startFadingEdgeStrength,
            ),
            size =
            Size(
                startFadingEdgeStrength,
                this.size.height,
            ),
        )

        drawRect(
            brush =
            Brush.horizontalGradient(
                colors =
                listOf(
                    Color.Transparent,
                    color,
                ),
                startX = size.width - endFadingEdgeStrength,
                endX = size.width,
            ),
            topLeft = Offset(x = size.width - endFadingEdgeStrength, y = 0f),
        )
      }
    }

fun Modifier.verticalFadingEdge(
  scrollState: ScrollState,
  length: Dp,
  edgeColor: Color? = null,
) =
    composed(
        debugInspectorInfo {
          name = "length"
          value = length
        },
    ) {
      val color = edgeColor ?: Color.Black.copy(alpha = 0.1f)

      drawWithContent {
        val lengthValue = length.toPx()
        val scrollFromTop by derivedStateOf { scrollState.value }
        val scrollFromBottom by derivedStateOf { scrollState.maxValue - scrollState.value }

        val topFadingEdgeStrength = lengthValue * (scrollFromTop / lengthValue).coerceAtMost(1f)

        val bottomFadingEdgeStrength =
            lengthValue * (scrollFromBottom / lengthValue).coerceAtMost(1f)

        drawContent()

        drawRect(
            brush =
            Brush.verticalGradient(
                colors =
                listOf(
                    color,
                    Color.Transparent,
                ),
                startY = 0f,
                endY = topFadingEdgeStrength,
            ),
            size = size.copy(height = topFadingEdgeStrength),
        )

        drawRect(
            brush =
            Brush.verticalGradient(
                colors =
                listOf(
                    Color.Transparent,
                    color,
                ),
                startY = size.height - bottomFadingEdgeStrength,
                endY = size.height,
            ),
            topLeft = Offset(x = 0f, y = size.height - bottomFadingEdgeStrength),
        )
      }
    }

fun Modifier.horizontalFadingEdge(
  lazyListState: LazyListState,
  length: Dp,
  edgeColor: Color? = null,
) =
    composed(
        debugInspectorInfo {
          name = "length"
          value = length
        },
    ) {
      val color = edgeColor ?: Color.Black.copy(alpha = 0.1f)

      drawWithContent {
        val startFadingEdgeStrength by derivedStateOf {
          lazyListState.layoutInfo
              .run {
                val firstItem = visibleItemsInfo.first()
                when {
                  firstItem.index >= 1 -> 1f
                  firstItem.offset == viewportStartOffset -> 0f
                  firstItem.offset < viewportStartOffset ->
                    firstItem.run { viewportStartOffset - firstItem.offset / size.toFloat() }

                  else -> 1f
                }
              }
              .coerceAtMost(1f) * length.value
        }
        val endFadingEdgeStrength by derivedStateOf {
          lazyListState.layoutInfo
              .run {
                val lastItem = visibleItemsInfo.last()
                when {
                  lastItem.index < totalItemsCount - 1 -> 1f
                  lastItem.offset + lastItem.size <= viewportEndOffset -> 0f
                  lastItem.offset + lastItem.size > viewportEndOffset ->
                    lastItem.run {
                      (size - (viewportEndOffset - lastItem.offset)) /
                          size.toFloat()
                    }

                  else -> 1f
                }
              }
              .coerceAtMost(1f) * length.value
        }

        drawContent()

        drawRect(
            brush =
            Brush.horizontalGradient(
                colors =
                listOf(
                    color,
                    Color.Transparent,
                ),
                startX = 0f,
                endX = startFadingEdgeStrength,
            ),
            size =
            Size(
                startFadingEdgeStrength,
                this.size.height,
            ),
        )

        drawRect(
            brush =
            Brush.horizontalGradient(
                colors =
                listOf(
                    Color.Transparent,
                    color,
                ),
                startX = size.width - endFadingEdgeStrength,
                endX = size.width,
            ),
            topLeft = Offset(x = size.width - endFadingEdgeStrength, y = 0f),
        )
      }
    }

fun Modifier.verticalFadingEdge(
  lazyListState: LazyListState,
  length: Dp,
  edgeColor: Color? = null,
) =
    composed(
        debugInspectorInfo {
          name = "length"
          value = length
        },
    ) {
      val color = edgeColor ?: Color.Black.copy(alpha = 0.1f)

      drawWithContent {
        val topFadingEdgeStrength by derivedStateOf {
          lazyListState.layoutInfo
              .run {
                val firstItem = visibleItemsInfo.first()
                when {
                  visibleItemsInfo.size in 0..1 -> 0f
                  firstItem.index > 0 -> 1f // Added
                  firstItem.offset == viewportStartOffset -> 0f
                  firstItem.offset < viewportStartOffset ->
                    firstItem.run { kotlin.math.abs(offset) / size.toFloat() }

                  else -> 1f
                }
              }
              .coerceAtMost(1f) * length.value
        }
        val bottomFadingEdgeStrength by derivedStateOf {
          lazyListState.layoutInfo
              .run {
                val lastItem = visibleItemsInfo.last()
                when {
                  visibleItemsInfo.size in 0..1 -> 0f
                  lastItem.index < totalItemsCount - 1 -> 1f // Added
                  lastItem.offset + lastItem.size <= viewportEndOffset -> 0f // added the <=
                  lastItem.offset + lastItem.size > viewportEndOffset ->
                    lastItem.run {
                      (size - (viewportEndOffset - offset)) /
                          size.toFloat() // Fixed the percentage computation
                    }

                  else -> 1f
                }
              }
              .coerceAtMost(1f) * length.value
        }

        drawContent()

        drawRect(
            brush =
            Brush.verticalGradient(
                colors =
                listOf(
                    color,
                    Color.Transparent,
                ),
                startY = 0f,
                endY = topFadingEdgeStrength,
            ),
            size = size.copy(height = topFadingEdgeStrength),
        )

        drawRect(
            brush =
            Brush.verticalGradient(
                colors =
                listOf(
                    Color.Transparent,
                    color,
                ),
                startY = size.height - bottomFadingEdgeStrength,
                endY = size.height,
            ),
            topLeft = Offset(x = 0f, y = size.height - bottomFadingEdgeStrength),
        )
      }
    }

fun Modifier.bottomBorder(strokeWidth: Dp, color: Color) = composed(
    factory = {
      val density = LocalDensity.current
      val strokeWidthPx = density.run { strokeWidth.toPx() }

      Modifier.drawBehind {
        val width = size.width
        val height = (size.height - strokeWidthPx/2) + 5

        drawLine(
            color = color,
            start = Offset(x = 0f, y = height),
            end = Offset(x = width , y = height),
            strokeWidth = strokeWidthPx
        )
      }
    }
)
