package org.codeberg.quecomet.oshi.ui.screens.upload.components

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.navigationBarsPadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowForward
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FilledIconButton
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.SheetState
import androidx.compose.material3.SheetValue
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import kotlinx.collections.immutable.persistentMapOf
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.model.UserMessage
import org.codeberg.quecomet.oshi.model.asString
import org.codeberg.quecomet.oshi.ui.components.NumericInputField
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.currentSpAsDp

val periodOptions =
    persistentMapOf<Int, UserMessage>(
        60 to UserMessage.from(R.string.period_1_hour),
        1440 to UserMessage.from(R.string.period_1_day),
        4320 to UserMessage.from(R.string.period_3_days),
        10080 to UserMessage.from(R.string.period_7_days),
        43200 to UserMessage.from(R.string.period_30_days),
        129600 to UserMessage.from(R.string.period_90_days),
    )

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PeriodSelectionInput(
    modifier: Modifier = Modifier,
    value: Int,
    onPeriodSelected: (Int) -> Unit,
    enabled: Boolean = true,
) {
  val scope = rememberCoroutineScope()
  val sheetState = rememberModalBottomSheetState(skipPartiallyExpanded = true)
  var showBottomSheet by remember { mutableStateOf(false) }

  val toggleModalBottomSheetState: () -> Unit = {
    scope.launch {
      if (showBottomSheet) {
        sheetState.hide()
        showBottomSheet = false
      } else {
        showBottomSheet = true
      }
    }
  }

  LaunchedEffect(showBottomSheet) {
    if (showBottomSheet) {
      sheetState.show()
    }
  }

  Surface(
      onClick = { toggleModalBottomSheetState() },
      enabled = enabled,
      modifier = modifier,
      shape = MaterialTheme.shapes.small,
  ) {
    Column(Modifier.width(IntrinsicSize.Max).padding(vertical = dimensionResource(R.dimen.padding_xs), horizontal =dimensionResource(R.dimen.padding_lg) )) {
      Text(
          stringResource(R.string.label_expiration_period),
          style = MaterialTheme.typography.labelSmall)
      Row(
          verticalAlignment = Alignment.CenterVertically,
          horizontalArrangement = Arrangement.SpaceBetween,
          modifier = Modifier.fillMaxWidth().padding(end = 5.dp),
      ) {
        Text(periodOptions[value]?.asString() ?: stringResource(R.string.minutes, value))
        Icon(
            imageVector = Icons.Default.ArrowDropDown,
            contentDescription = null,
            modifier = Modifier.size(currentSpAsDp().times(1.5f)),
        )
      }
      HorizontalDivider(Modifier.padding(end = currentSpAsDp().times(0.4f)))
    }
  }

  if (showBottomSheet) {
    PeriodSelectionSheet(
        sheetState = sheetState,
        toggleModalBottomSheetState = toggleModalBottomSheetState,
        onPeriodSelected = onPeriodSelected,
        value = value,
    )
  }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun PeriodSelectionSheet(
    sheetState: SheetState = rememberModalBottomSheetState(),
    toggleModalBottomSheetState: () -> Unit,
    onPeriodSelected: (Int) -> Unit,
    value: Int,
) {

  ModalBottomSheet(
      onDismissRequest = toggleModalBottomSheetState,
      sheetState = sheetState,
  ) {
    LazyColumn(
        modifier = Modifier.navigationBarsPadding(),
    ) {
      item {
        var customPeriod by remember { mutableStateOf(value.toString()) }
        Row(
            modifier = Modifier.padding(dimensionResource(R.dimen.padding_lg)),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_lg)),
        ) {
          NumericInputField(
              value = customPeriod,
              onValueChange = { customPeriod = it },
              label = { Text(stringResource(R.string.custom_expiration_value)) },
              minValue = 1,
              maxValue = 129600,
              modifier = Modifier.weight(1f, true),
              trailingIcon = {
                Box(Modifier.padding(end = 5.dp)) {
                  FilledIconButton(
                      onClick = {
                        customPeriod.toIntOrNull()?.let {
                          onPeriodSelected(it)
                          toggleModalBottomSheetState()
                        }
                      },
                  ) {
                    Icon(
                        imageVector = Icons.AutoMirrored.Default.ArrowForward,
                        contentDescription = stringResource(R.string.use))
                  }
                }

              },
              shape = MaterialTheme.shapes.extraLarge,
          )
        }
      }
      items(items = periodOptions.entries.toList(), key = { it.key }) {
        Text(
            text = it.value.asString(),
            modifier =
                Modifier.fillMaxWidth()
                    .clickable {
                      onPeriodSelected(it.key)
                      toggleModalBottomSheetState()
                    }
                    .padding(
                        horizontal = dimensionResource(R.dimen.padding_xl),
                        vertical = dimensionResource(R.dimen.padding_lg),
                    ),
            fontSize = 18.sp,
        )
      }
    }
  }
}

@OptIn(ExperimentalMaterial3Api::class)
@Preview("sheet")
@Composable
fun PeriodSelectionSheetPreview() {
  OshiTheme {
    val density = LocalDensity.current
    PeriodSelectionSheet(
        toggleModalBottomSheetState = {},
        onPeriodSelected = {},
        sheetState =
            SheetState(
                skipPartiallyExpanded = false,
                initialValue = SheetValue.Expanded,
                density = density,
            ),
        value = 1440,
    )
  }
}

@Preview("input")
@Composable
fun PeriodSelectionInputPreview() {
  OshiTheme {
    Surface {
      PeriodSelectionInput(
          value = 1440,
          onPeriodSelected = {},
      )
    }
  }
}
