package org.codeberg.quecomet.oshi.ui.screens.upload.components

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.navigationBarsPadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.widthIn
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material.icons.filled.DeleteForever
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Search
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.SearchBar
import androidx.compose.material3.SheetState
import androidx.compose.material3.SheetValue
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.paging.LoadState
import androidx.paging.PagingData
import androidx.paging.compose.collectAsLazyPagingItems
import androidx.paging.compose.itemKey
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.OshiInstanceRepositoryMock
import org.codeberg.quecomet.oshi.data.room.OshiInstance
import org.codeberg.quecomet.oshi.ui.components.Card
import org.codeberg.quecomet.oshi.ui.components.ConfirmDeleteDialog
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.currentSpAsDp

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun OshiInstanceSelectDropdown(
    modifier: Modifier = Modifier,
    value: OshiInstance?,
    searchTerm: String = "",
    onDismissRequest: () -> Unit = {},
    onSearchTermUpdated: (String) -> Unit = {},
    onInstanceSelected: (OshiInstance) -> Unit,
    onInstanceDelete: suspend (OshiInstance) -> Unit,
    onInstanceSave: suspend (OshiInstance, isEdit: Boolean) -> Unit,
    oshiInstanceList: Flow<PagingData<OshiInstance>>,
    enabled: Boolean = true,
    forceShow: Boolean = false,
) {
  val scope = rememberCoroutineScope()
  val sheetState = rememberModalBottomSheetState()
  var shouldShowSheet by remember { mutableStateOf(false) }

  val localOnDismissRequest: () -> Unit = {
    onDismissRequest()
    scope.launch { sheetState.hide() }
    shouldShowSheet = false
  }

  LaunchedEffect(forceShow) {
    if (sheetState.currentValue == SheetValue.Hidden && forceShow) {
      shouldShowSheet = true
      sheetState.show()
    }
  }

  TextButton(
      onClick = {
        shouldShowSheet = true
        scope.launch { sheetState.show() }
      },
      enabled = enabled,
      modifier = modifier,
  ) {
    Column(Modifier.width(IntrinsicSize.Max)) {
      Row(verticalAlignment = Alignment.CenterVertically) {
        if (value == null) {
          Text(stringResource(R.string.upload_select_instance))
        } else {
          Text(
              value.label.toString(),
              modifier = Modifier.widthIn(0.dp, 200.dp),
              maxLines = 1,
              overflow = TextOverflow.Ellipsis)
        }
        Icon(
            imageVector = Icons.Default.ArrowDropDown,
            contentDescription = null,
            modifier = Modifier.size(currentSpAsDp().times(1.5f)),
        )
      }
      HorizontalDivider(Modifier.padding(end = currentSpAsDp().times(0.4f)))
    }
  }

  if (shouldShowSheet) {
    InstanceSelectionSheet(
        sheetState = sheetState,
        onDismissRequest = localOnDismissRequest,
        searchTerm = searchTerm,
        onSearchTermUpdated = onSearchTermUpdated,
        onInstanceSelected = onInstanceSelected,
        oshiInstanceList = oshiInstanceList,
        onInstanceDelete = onInstanceDelete,
        onInstanceSave = onInstanceSave,
    )
  }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun InstanceSelectionSheet(
    sheetState: SheetState = rememberModalBottomSheetState(),
    onDismissRequest: () -> Unit,
    searchTerm: String = "",
    onSearchTermUpdated: (String) -> Unit = {},
    onInstanceSelected: (OshiInstance) -> Unit,
    onInstanceDelete: suspend (OshiInstance) -> Unit,
    onInstanceSave: suspend (OshiInstance, isEdit: Boolean) -> Unit,
    oshiInstanceList: Flow<PagingData<OshiInstance>>,
) {
  val instances = oshiInstanceList.collectAsLazyPagingItems()
  val scope = rememberCoroutineScope()

  var editInstance by remember { mutableStateOf<OshiInstance?>(null) }
  var showCreateEditDialog by remember { mutableStateOf(false) }
  var showDeleteConfirmDialog by remember { mutableStateOf<OshiInstance?>(null) }

  ModalBottomSheet(
      onDismissRequest = onDismissRequest,
      sheetState = sheetState,
  ) {
    Box(
        modifier = Modifier.navigationBarsPadding().weight(1f),
    ) {
      when (instances.loadState.refresh) {
        is LoadState.Error ->
            Card {
              Column(Modifier.padding(vertical = 70.dp, horizontal = 20.dp)) {
                Text(stringResource(R.string.unknown_error))
              }
            }
        else -> {
          LazyColumn(
              horizontalAlignment = Alignment.CenterHorizontally,
              verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.page_padding)),
          ) {
            item {
              SearchBar(
                  modifier =
                      Modifier.fillMaxWidth()
                          .padding(horizontal = dimensionResource(R.dimen.page_padding)),
                  query = searchTerm,
                  windowInsets = WindowInsets(0.dp, 0.dp, 0.dp, 0.dp),
                  onSearch = {},
                  onQueryChange = { onSearchTermUpdated(it) },
                  active = false,
                  onActiveChange = {},
                  placeholder = { Text(text = stringResource(R.string.search_placeholder)) },
                  leadingIcon = {
                    Icon(imageVector = Icons.Default.Search, contentDescription = null)
                  },
                  trailingIcon = {
                    TextButton(
                        onClick = {
                          editInstance = null
                          showCreateEditDialog = true
                        }) {
                          Row(verticalAlignment = Alignment.CenterVertically) {
                            Text(stringResource(R.string.add))
                            Icon(imageVector = Icons.Default.Add, contentDescription = null)
                          }
                        }
                  }) {}
            }

            if (instances.loadState.refresh is LoadState.Loading) {
              item {
                CircularProgressIndicator(
                    modifier = Modifier.align(Alignment.Center).size(50.dp),
                )
              }
            }

            if (instances.itemCount == 0) {
              item {
                Column(
                    Modifier.fillMaxWidth()
                        .padding(
                            horizontal = dimensionResource(R.dimen.page_padding),
                            vertical = 70.dp,
                        ),
                    horizontalAlignment = Alignment.CenterHorizontally,
                    verticalArrangement = Arrangement.Center) {
                      Text(
                          if (searchTerm.isBlank())
                              stringResource(R.string.notice_empty_oshi_instances)
                          else stringResource(R.string.search_had_no_result),
                          fontSize = 20.sp,
                          textAlign = TextAlign.Center,
                      )
                    }
              }
            }

            items(
                count = instances.itemCount,
                key = instances.itemKey { it.id },
            ) { index ->
              val item = instances[index]
              if (item != null) {
                Row(
                    Modifier.fillMaxWidth()
                        .clickable {
                          onInstanceSelected(item)
                          onDismissRequest()
                        }
                        .padding(
                            top = dimensionResource(R.dimen.padding_xs),
                            bottom = dimensionResource(R.dimen.padding_xs),
                            start = dimensionResource(R.dimen.padding_xl),
                            end = dimensionResource(R.dimen.padding_sm),
                        ),
                    horizontalArrangement = Arrangement.spacedBy(5.dp),
                    verticalAlignment = Alignment.CenterVertically,
                ) {
                  Text(
                      text = item.label.toString(),
                      fontSize = 18.sp,
                      modifier = if (item.label.isBlank()) Modifier.weight(1f) else Modifier,
                  )
                  if (item.label.isNotBlank()) {
                    Text(
                        text = "(${item.host})",
                        color = MaterialTheme.colorScheme.onSurfaceVariant,
                        fontSize = MaterialTheme.typography.bodySmall.fontSize,
                        modifier = Modifier.weight(1f),
                        overflow = TextOverflow.Ellipsis,
                        maxLines = 1,
                    )
                  }
                  IconButton(
                      onClick = {
                        editInstance = item
                        showCreateEditDialog = true
                      }) {
                        Icon(
                            imageVector = Icons.Default.Edit,
                            contentDescription = stringResource(R.string.title_edit_instance))
                      }
                  IconButton(onClick = { scope.launch { showDeleteConfirmDialog = item } }) {
                    Icon(
                        imageVector = Icons.Default.DeleteForever,
                        contentDescription = stringResource(R.string.delete_instance))
                  }
                }
              }
            }

            item {
              if (instances.loadState.append == LoadState.Loading) {
                CircularProgressIndicator(modifier = Modifier.padding(16.dp))
              }
            }
          }
        }
      }
    }
  }

  if (showDeleteConfirmDialog != null) {
    val instance = showDeleteConfirmDialog!!
    ConfirmDeleteDialog(
        itemLabel = instance.toString(),
        onConfirm = {
          scope.launch {
            onInstanceDelete(instance)
            showDeleteConfirmDialog = null
          }
        },
        onDismissRequest = { showDeleteConfirmDialog = null },
        confirmMessage = R.string.instance_delete_dialog_confirmation,
    )
  }

  OshiInstanceCreateEditDialog(
      show = showCreateEditDialog,
      editInstance = editInstance,
      onInstanceSave = onInstanceSave,
      onDismissRequest = { showCreateEditDialog = false })
}

@OptIn(ExperimentalMaterial3Api::class)
@Preview("sheet")
@Composable
fun OshiInstanceSelectSheetPreview() {
  OshiTheme {
    val density = LocalDensity.current
    InstanceSelectionSheet(
        onInstanceSelected = {},
        sheetState =
            SheetState(
                skipPartiallyExpanded = false,
                initialValue = SheetValue.Expanded,
                density = density),
        oshiInstanceList = OshiInstanceRepositoryMock().getInstanceList(),
        onInstanceDelete = {},
        onInstanceSave = { _, _ -> },
        onDismissRequest = {},
    )
  }
}

@OptIn(ExperimentalMaterial3Api::class)
@Preview("sheet - empty")
@Composable
fun OshiInstanceSelectSheetEmptyPreview() {
  OshiTheme {
    val density = LocalDensity.current
    InstanceSelectionSheet(
        onInstanceSelected = {},
        sheetState =
            SheetState(
                skipPartiallyExpanded = false,
                initialValue = SheetValue.Expanded,
                density = density),
        oshiInstanceList = OshiInstanceRepositoryMock().getEmptyInstanceList(),
        onInstanceDelete = {},
        onInstanceSave = { _, _ -> },
        onDismissRequest = {},
    )
  }
}

@Preview("input")
@Composable
fun OshiInstanceSelectInputPreview() {
  OshiTheme {
    Surface {
      OshiInstanceSelectDropdown(
          value = OshiInstanceRepositoryMock.instances[0],
          onInstanceSelected = {},
          onInstanceDelete = {},
          onInstanceSave = { _, _ -> },
          oshiInstanceList = OshiInstanceRepositoryMock().getInstanceList(),
      )
    }
  }
}
