package org.codeberg.quecomet.oshi.ui.screens.upload.components

import android.database.sqlite.SQLiteConstraintException
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Button
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import io.ktor.http.URLProtocol
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.data.OshiInstanceRepositoryMock
import org.codeberg.quecomet.oshi.data.room.OshiInstance
import org.codeberg.quecomet.oshi.model.UserMessage
import org.codeberg.quecomet.oshi.model.asString
import org.codeberg.quecomet.oshi.ui.components.BasicDialog
import org.codeberg.quecomet.oshi.ui.components.CheckBoxRow
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.verticalFadingEdge

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun OshiInstanceCreateEditDialog(
    modifier: Modifier = Modifier,
    show: Boolean = true,
    editInstance: OshiInstance? = null,
    onInstanceSave: suspend (OshiInstance, isEdit: Boolean) -> Unit,
    onDismissRequest: () -> Unit,
) {
  val scope = rememberCoroutineScope()
  if (show) {
    var working by remember { mutableStateOf(false) }
    var instanceDraft by remember {
      mutableStateOf(
          editInstance?.copy() ?: OshiInstance(host = "", protocol = URLProtocol.HTTPS, label = ""))
    }
    var saveError by remember { mutableStateOf<UserMessage?>(null) }
    val scrollState = rememberScrollState()

    BasicDialog(
        onDismissRequest = onDismissRequest,
    ) {
      Column(
          horizontalAlignment = Alignment.CenterHorizontally,
          verticalArrangement = Arrangement.spacedBy(5.dp),
          modifier =
              modifier
                  .verticalFadingEdge(
                      scrollState = scrollState,
                      length = 30.dp,
                      edgeColor = CardDefaults.cardColors().containerColor)
                  .verticalScroll(scrollState)
                  .padding(dimensionResource(R.dimen.padding_xl)),
      ) {
        Text(
            text =
                if (editInstance == null) stringResource(R.string.title_add_instance)
                else stringResource(R.string.title_edit_instance),
            style = MaterialTheme.typography.titleLarge,
        )

        OutlinedTextField(
            modifier = Modifier.fillMaxWidth(),
            value = instanceDraft.label,
            textStyle = MaterialTheme.typography.bodyLarge,
            onValueChange = { instanceDraft = instanceDraft.copy(label = it) },
            label = { Text(stringResource(R.string.label)) },
            placeholder = { Text(stringResource(R.string.instance_label_placeholder)) },
            enabled = !working,
            singleLine = true,
            maxLines = 1,
        )

        OutlinedTextField(
            modifier = Modifier.fillMaxWidth(),
            value = instanceDraft.host,
            textStyle = MaterialTheme.typography.bodyMedium,
            onValueChange = { instanceDraft = instanceDraft.copy(host = it) },
            label = { Text(stringResource(R.string.hostname)) },
            enabled = !working,
            singleLine = true,
            maxLines = 1,
            placeholder = { Text(stringResource(R.string.instance_host_placeholder)) },
            supportingText = {
              if (saveError != null) {
                Text(text = saveError!!.asString(), color = MaterialTheme.colorScheme.error)
              }
            },
        )

        CheckBoxRow(
            modifier = Modifier.fillMaxWidth(),
            checked = instanceDraft.protocol == URLProtocol.HTTPS,
            onCheckedChange = {
              instanceDraft =
                  if (it) instanceDraft.copy(protocol = URLProtocol.HTTPS)
                  else instanceDraft.copy(protocol = URLProtocol.HTTP)
            },
        ) {
          Text(
              stringResource(R.string.instance_protocol_https),
              style = MaterialTheme.typography.bodyMedium)
        }

        var showAdvancedOptions by remember { mutableStateOf(false) }

        Row(Modifier.fillMaxWidth()) {
          TextButton(onClick = { showAdvancedOptions = !showAdvancedOptions }) {
            if (!showAdvancedOptions) {
              Text(stringResource(R.string.show_advanced_settings))
            } else {
              Text(stringResource(R.string.hide_advanced_settings))
            }
          }
        }

        AnimatedVisibility(visible = showAdvancedOptions) {
          Column(
              horizontalAlignment = Alignment.CenterHorizontally,
              verticalArrangement = Arrangement.spacedBy(5.dp),
          ) {
            OutlinedTextField(
                modifier = Modifier.fillMaxWidth(),
                value = instanceDraft.sha1Fingerprint ?: "",
                textStyle = MaterialTheme.typography.bodyMedium,
                onValueChange = {
                  instanceDraft =
                      instanceDraft.copy(
                          sha1Fingerprint = it,
                          fingerprintConfirmedByUser = it.isNotBlank(),
                      )
                },
                label = { Text(stringResource(R.string.instance_ssl_sha1_fingerprint)) },
                enabled = !working,
                singleLine = true,
                maxLines = 1,
                placeholder = {
                  Text(stringResource(R.string.instance_ssl_sha1_fingerprint_placeholder))
                },
            )
            OutlinedTextField(
                modifier = Modifier.fillMaxWidth(),
                value = instanceDraft.sha256Fingerprint ?: "",
                textStyle = MaterialTheme.typography.bodyMedium,
                onValueChange = {
                  instanceDraft =
                      instanceDraft.copy(
                          sha256Fingerprint = it,
                          fingerprintConfirmedByUser = it.isNotBlank(),
                      )
                },
                label = { Text(stringResource(R.string.instance_ssl_sha256_fingerprint)) },
                enabled = !working,
                singleLine = true,
                maxLines = 1,
                placeholder = {
                  Text(stringResource(R.string.instance_ssl_sha256_fingerprint_placeholder))
                },
            )

            Spacer(Modifier.height(10.dp))
            HorizontalDivider(
                thickness = 1.dp, color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.3f))
            Spacer(Modifier.height(10.dp))

            OutlinedTextField(
                modifier = Modifier.fillMaxWidth(),
                value = instanceDraft.alternateHost ?: "",
                textStyle = MaterialTheme.typography.bodyMedium,
                onValueChange = { instanceDraft = instanceDraft.copy(alternateHost = it) },
                label = { Text(stringResource(R.string.alternate_hostname)) },
                enabled = !working,
                maxLines = 1,
                singleLine = true,
                placeholder = {
                  Text(stringResource(R.string.instance_alternate_host_placeholder))
                },
                supportingText = {
                  Text(
                      stringResource(R.string.instance_alternate_host_supporting_text),
                      style = MaterialTheme.typography.bodySmall,
                  )
                })

            CheckBoxRow(
                modifier = Modifier.fillMaxWidth(),
                checked = instanceDraft.alternateProtocol == URLProtocol.HTTPS,
                onCheckedChange = {
                  instanceDraft =
                      if (it) instanceDraft.copy(alternateProtocol = URLProtocol.HTTPS)
                      else instanceDraft.copy(alternateProtocol = URLProtocol.HTTP)
                },
            ) {
              Text(
                  stringResource(R.string.instance_alternate_protocol_https),
                  style = MaterialTheme.typography.bodyMedium)
            }
          }
        }

        Spacer(Modifier)

        Row(horizontalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_md))) {
          TextButton(onClick = { onDismissRequest() }) { Text(stringResource(R.string.cancel)) }
          Button(
              onClick = {
                if (working) return@Button
                working = true
                saveError = null
                scope
                    .launch {
                      try {
                        onInstanceSave(instanceDraft, editInstance != null)
                        onDismissRequest()
                      } catch (e: Throwable) {
                        saveError =
                            if (e is SQLiteConstraintException) {
                              UserMessage.from(R.string.error_instead_already_exists)
                            } else {
                              UserMessage.from(e.message ?: e.toString())
                            }
                      }
                    }
                    .invokeOnCompletion { working = false }
              },
              enabled = !working) {
                if (working) {
                  CircularProgressIndicator()
                } else {
                  if (editInstance == null) {
                    Text(stringResource(R.string.add))
                  } else {
                    Text(stringResource(R.string.label_save))
                  }
                }
              }
        }
      }
    }
  }
}

@Preview("Create Dialog")
@Composable
fun OshiInstanceCreateDialogPreview() {
  OshiTheme {
    Surface {
      OshiInstanceCreateEditDialog(
          onInstanceSave = { _, _ -> },
          onDismissRequest = {},
      )
    }
  }
}

@Preview("Edit Dialog")
@Composable
fun OshiInstanceEditDialogPreview() {
  OshiTheme {
    Surface {
      OshiInstanceCreateEditDialog(
          editInstance = OshiInstanceRepositoryMock.instances[0],
          onInstanceSave = { _, _ -> },
          onDismissRequest = {},
      )
    }
  }
}
