package org.codeberg.quecomet.oshi.ui.screens.upload

import android.Manifest
import android.os.Build
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.paging.LoadState
import androidx.paging.compose.collectAsLazyPagingItems
import androidx.paging.compose.itemKey
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.SSLCheckMode
import org.codeberg.quecomet.oshi.ShareEvent
import org.codeberg.quecomet.oshi.ShareIntentHandler
import org.codeberg.quecomet.oshi.data.OshiInstanceRepositoryMock
import org.codeberg.quecomet.oshi.data.UploadWorkRepositoryMock
import org.codeberg.quecomet.oshi.data.UserSettingsRepositoryMock
import org.codeberg.quecomet.oshi.ui.components.Card
import org.codeberg.quecomet.oshi.ui.components.CheckBoxRow
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.ui.components.WorkingDialog
import org.codeberg.quecomet.oshi.ui.navigation.SecondaryDestinations
import org.codeberg.quecomet.oshi.ui.screens.upload.components.ConfirmFingerprintDialog
import org.codeberg.quecomet.oshi.ui.screens.upload.components.FileSelector
import org.codeberg.quecomet.oshi.ui.screens.upload.components.NotificationPermissionDialog
import org.codeberg.quecomet.oshi.ui.screens.upload.components.OshiInstanceSelectDropdown
import org.codeberg.quecomet.oshi.ui.screens.upload.components.PeriodSelectionInput
import org.codeberg.quecomet.oshi.ui.screens.upload.components.UploadWorkItem
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.noRippleClickable
import org.codeberg.quecomet.oshi.utils.createNotificationChannels

@OptIn(ExperimentalMaterial3Api::class, ExperimentalFoundationApi::class)
@Composable
fun Upload(
    modifier: Modifier = Modifier,
    viewModel: UploadViewModel,
    onNavigateToRoute: (String) -> Unit,
) {
  val context = LocalContext.current
  val focusManager = LocalFocusManager.current
  val uploadItems = viewModel.uploadWorkPagingDataFlow.collectAsLazyPagingItems()
  val selectedOshiInstance by viewModel.userSelectedOshiInstance.collectAsStateWithLifecycle()
  val userSettings by viewModel.userSettings.collectAsStateWithLifecycle()
  val searchTerm by viewModel.oshiInstanceSearchTerm.collectAsStateWithLifecycle()
  val event by viewModel.shareIntentHandler.shareEvent.collectAsStateWithLifecycle()

  val launcher =
      rememberLauncherForActivityResult(ActivityResultContracts.RequestPermission()) {
        if (it) {
          createNotificationChannels(context)
        }
      }

  LaunchedEffect(event) {
    when (val currentEvent = event) {
      is ShareEvent.HandleShareData -> {
        viewModel.handleShareIntent(currentEvent.intent, context)
      }
      ShareEvent.None -> Unit
    }
    viewModel.shareIntentHandler.consumeEvent()
  }

  LaunchedEffect(userSettings, selectedOshiInstance) {
    if (selectedOshiInstance != null &&
        userSettings != null &&
        userSettings?.sslCheckMode != SSLCheckMode.CERTIFICATE_ONLY) {
      viewModel.confirmFingerprint(selectedOshiInstance)
    }
  }

  Scaffold(
      snackbarHost = { SnackbarHost(hostState = SnackbarManager.snackbarHostState) },
      topBar = {
        CenterAlignedTopAppBar(
            colors = TopAppBarDefaults.centerAlignedTopAppBarColors(),
            title = {
              OshiInstanceSelectDropdown(
                  value = selectedOshiInstance,
                  searchTerm = searchTerm,
                  onSearchTermUpdated = viewModel::updateOshiInstanceSearchTerm,
                  onInstanceSelected = { viewModel.setSelectedOshiInstance(it) },
                  onInstanceDelete = { viewModel.onDeleteOshiInstance(it) },
                  onInstanceSave = { instance, isEdit -> viewModel.saveInstance(instance, isEdit) },
                  oshiInstanceList = viewModel.oshiInstancesPagingDataFlow,
                  forceShow = viewModel.workUUIDRequestingInstanceChange.isNotBlank(),
                  onDismissRequest = { viewModel.onChangeInstanceRequestCancelled() })
            },
            navigationIcon = {
              IconButton(onClick = { onNavigateToRoute(SecondaryDestinations.SETTINGS_ROUTE) }) {
                Icon(
                    modifier = Modifier.size(32.dp),
                    imageVector = Icons.Default.Settings,
                    contentDescription = null,
                )
              }
            },
        )
      },
  ) { padding ->
    Column(
        modifier =
            modifier
                .padding(padding)
                .padding(horizontal = dimensionResource(R.dimen.page_padding))
                .noRippleClickable() { focusManager.clearFocus() },
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_md)),
    ) {
      if (viewModel.checkingFingerprint) {
        WorkingDialog(onDismissRequest = { viewModel.cancelFingerprintCheck() }) {
          Text(
              stringResource(R.string.checking_fingerprint),
              style = MaterialTheme.typography.titleLarge,
          )
        }
      }

      if (viewModel.showNotificationPermissionRequest) {
        NotificationPermissionDialog {
          viewModel.updateShowNotificationPermissionRequest(false)
          if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            launcher.launch(Manifest.permission.POST_NOTIFICATIONS)
          }
        }
      }

      Column(
          modifier = Modifier.padding(top = dimensionResource(R.dimen.page_padding)),
      ) {
        Row(modifier = Modifier.padding(start = 8.dp)) {
          FileSelector(
              viewModel.selectedFiles,
              onFilesSelected = { viewModel.updateSelectedFiles(it) },
              onClearSelection = { viewModel.updateSelectedFiles(emptyList()) },
              onUploadRequested = { viewModel.uploadSelectedFiles() })
        }

        Spacer(Modifier.height(12.dp))
        Row(horizontalArrangement = Arrangement.SpaceBetween, modifier = Modifier.fillMaxWidth()) {
          CheckBoxRow(
              checked = userSettings?.uploadDestroyAfterDl == true,
              onCheckedChange = { viewModel.updateUploadDestroyAfterDL(it) },
              modifier = Modifier.weight(1f),
          ) {
            Text(
                stringResource(R.string.destroy_after_download),
                style = MaterialTheme.typography.bodyMedium)
          }
          CheckBoxRow(
              checked = userSettings?.uploadRandomizeName == true,
              onCheckedChange = { viewModel.updateUploadRandomizeName(it) },
              modifier = Modifier.weight(1f),
          ) {
            Text(
                stringResource(R.string.randomize_name),
                style = MaterialTheme.typography.bodyMedium)
          }
        }
        Row(horizontalArrangement = Arrangement.SpaceBetween, modifier = Modifier.fillMaxWidth()) {
          CheckBoxRow(
              checked = userSettings?.uploadShortenUrl == true,
              onCheckedChange = { viewModel.updateUploadShortenUrl(it) },
              modifier = Modifier.weight(1f),
          ) {
            Text(stringResource(R.string.short_url), style = MaterialTheme.typography.bodyMedium)
          }

          PeriodSelectionInput(
              value = userSettings?.uploadExpiration ?: 1440,
              onPeriodSelected = { viewModel.updateUploadExpiration(it) },
              modifier = Modifier.weight(1f),
          )
        }
      }

      HorizontalDivider()

      Spacer(Modifier.height(dimensionResource(R.dimen.page_padding)))

      Box(
          modifier = Modifier.weight(1f),
      ) {
        when (uploadItems.loadState.refresh) {
          is LoadState.Error ->
              Card {
                Column(Modifier.padding(vertical = 70.dp, horizontal = 20.dp)) {
                  Text(stringResource(R.string.unknown_error))
                }
              }
          else -> {
            if (uploadItems.itemCount == 0) {
              Column(
                  Modifier.fillMaxSize(1f)
                      .padding(
                          horizontal = dimensionResource(R.dimen.page_padding),
                          vertical = 70.dp,
                      ),
                  horizontalAlignment = Alignment.CenterHorizontally,
                  verticalArrangement = Arrangement.Center) {
                    Text(
                        stringResource(R.string.notice_empty_active_upload),
                        fontSize = 22.sp,
                        textAlign = TextAlign.Center,
                    )
                  }
            } else {
              LazyColumn(
                  horizontalAlignment = Alignment.CenterHorizontally,
                  verticalArrangement =
                      Arrangement.spacedBy(dimensionResource(R.dimen.page_padding)),
              ) {
                if (uploadItems.loadState.refresh is LoadState.Loading) {
                  item {
                    CircularProgressIndicator(
                        modifier =
                            Modifier.align(Alignment.Center).size(50.dp).animateItemPlacement(),
                    )
                  }
                }

                items(
                    count = uploadItems.itemCount,
                    key = uploadItems.itemKey { it.uploadWork.workUUID },
                ) { index ->
                  val item = uploadItems[index]
                  if (item != null) {
                    UploadWorkItem(
                        modifier = Modifier.animateItemPlacement(),
                        uploadWorkAndOshiInstance = item,
                        workInfoFlow = viewModel.getUploadWorkWorkInfo(item.uploadWork.workUUID),
                        onRetry = viewModel::retryUploadWork,
                        onCancel = viewModel::cancelUploadWork,
                        onDelete = { viewModel.deleteUploadWork(it, context) },
                        onChangeInstanceRequest = { viewModel.onChangeInstanceRequest(it) })
                  }
                }

                item {
                  if (uploadItems.loadState.append == LoadState.Loading) {
                    CircularProgressIndicator(
                        modifier = Modifier.padding(16.dp).animateItemPlacement(),
                    )
                  }
                }
              }
            }
          }
        }
      }

      ConfirmFingerprintDialog(
          show = viewModel.instanceToConfirmSSLFingerprint != null,
          oshiInstance = viewModel.instanceToConfirmSSLFingerprint,
          onConfirm = viewModel::onInstanceSSLConfirmed,
          onDismissRequest = { viewModel.updateInstanceToConfirmSSLFingerprint(null) })
    }
  }
}

@Preview("default")
@Composable
fun UploadPreview() {
  val viewModel =
      UploadViewModel(
          SavedStateHandle(),
          uploadWorkRepository = UploadWorkRepositoryMock(),
          oshiInstanceRepository = OshiInstanceRepositoryMock(),
          userSettingsRepository = UserSettingsRepositoryMock(),
          shareIntentHandler = ShareIntentHandler(),
      )
  OshiTheme(darkTheme = true) {
    Upload(
        viewModel = viewModel,
        onNavigateToRoute = {},
    )
  }
}

@Preview("default - empty")
@Composable
fun UploadEmptyPreview() {
  val viewModel =
      UploadViewModel(
          SavedStateHandle(),
          uploadWorkRepository = UploadWorkRepositoryMock(false),
          oshiInstanceRepository = OshiInstanceRepositoryMock(),
          userSettingsRepository = UserSettingsRepositoryMock(),
          shareIntentHandler = ShareIntentHandler(),
      )
  OshiTheme(darkTheme = true) {
    Upload(
        viewModel = viewModel,
        onNavigateToRoute = {},
    )
  }
}
