package org.codeberg.quecomet.oshi.ui.screens.settings

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.exclude
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.ime
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.navigationBars
import androidx.compose.foundation.layout.navigationBarsPadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Button
import androidx.compose.material3.Card
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.ScaffoldDefaults
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import kotlinx.collections.immutable.persistentListOf
import org.codeberg.quecomet.oshi.BuildConfig
import org.codeberg.quecomet.oshi.PreferredProxyType
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.SSLCheckMode
import org.codeberg.quecomet.oshi.data.UserSettingsRepositoryMock
import org.codeberg.quecomet.oshi.ui.components.NumericInputField
import org.codeberg.quecomet.oshi.ui.components.RadioGroup
import org.codeberg.quecomet.oshi.ui.components.RadioGroupRow
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.ui.components.UpBtn
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.ui.utils.noRippleClickable
import org.codeberg.quecomet.oshi.ui.utils.verticalFadingEdge

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun Settings(viewModel: SettingsViewModel, upPress: () -> Unit, modifier: Modifier = Modifier) {
  val focusManager = LocalFocusManager.current
  val lifecycleOwner = LocalLifecycleOwner.current
  val lifecycleState by lifecycleOwner.lifecycle.currentStateFlow.collectAsStateWithLifecycle()
  val scrollState = rememberScrollState()
  val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())

  LaunchedEffect(lifecycleState) {
    when (lifecycleState) {
      Lifecycle.State.STARTED, Lifecycle.State.RESUMED -> {
        viewModel.reloadSettings()
      }

      else -> {}
    }
  }

  Scaffold(
      modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
      snackbarHost = {
        SnackbarHost(
            hostState = SnackbarManager.snackbarHostState,
            modifier = Modifier
                .navigationBarsPadding()
                .imePadding(),
        )
      },
      topBar = {
        TopAppBar(
            scrollBehavior = scrollBehavior,
            colors = TopAppBarDefaults.topAppBarColors(
                scrolledContainerColor = MaterialTheme.colorScheme.inverseOnSurface,
            ),
            title = { Text(stringResource(R.string.settings)) },
            navigationIcon = { UpBtn(upPress) },
        )
      },
      contentWindowInsets = ScaffoldDefaults.contentWindowInsets.exclude(WindowInsets.navigationBars)
          .exclude(WindowInsets.ime),
  ) { padding ->
    Column(
        modifier = modifier
            .padding(padding)
            .navigationBarsPadding()
            .imePadding()
            .padding(
                horizontal = dimensionResource(R.dimen.page_padding),
            )
            .verticalFadingEdge(scrollState, dimensionResource(R.dimen.fading_edge))
            .verticalScroll(scrollState)
            .noRippleClickable() { focusManager.clearFocus() },
        verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_md)),
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {

      // region proxy
      Card {
        Column(
            modifier = Modifier.padding(dimensionResource(R.dimen.padding_xl)),
            verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_sm)),
        ) {
          Text(text = stringResource(R.string.label_proxy_settings), fontSize = 20.sp)

          SettingItemSwitch(
              text = stringResource(R.string.label_is_proxy_enabled),
              checked = viewModel.isProxyEnabledDraft,
              onCheckedChange = { viewModel.updateIsProxyEnabledDraft(it) },
          )
          OutlinedTextField(
              modifier = Modifier.fillMaxWidth(),
              enabled = viewModel.isProxyEnabledDraft,
              value = viewModel.proxyHostDraft,
              onValueChange = { viewModel.updateProxyHostDraft(it) },
              label = { Text(stringResource(R.string.label_proxy_host)) },
          )
          NumericInputField(
              modifier = Modifier.fillMaxWidth(),
              enabled = viewModel.isProxyEnabledDraft,
              value = viewModel.proxyPortDraft,
              minValue = 0,
              maxValue = 65535,
              onValueChange = { viewModel.updateProxyPortDraft(it) },
              label = { Text(stringResource(R.string.label_proxy_port)) },
          )

          Spacer(Modifier.padding(bottom = 10.dp))

          Text(text = stringResource(R.string.label_preferred_proxy_type), fontSize = 20.sp)
          RadioGroupRow(
              enabled = viewModel.isProxyEnabledDraft,
              options = persistentListOf(PreferredProxyType.SOCKS5, PreferredProxyType.HTTP),
              selectedOption = viewModel.preferredProxyTypeDraft,
              onOptionSelected = { viewModel.updatePreferredProxyTypeDraft(it) },
              label = {
                Text(
                    when (it) {
                      PreferredProxyType.SOCKS5 -> stringResource(R.string.label_proxy_type_socks5)
                      PreferredProxyType.HTTP -> stringResource(R.string.label_proxy_type_http)
                      else -> stringResource(R.string.Unknown)
                    },
                )
              },
          )

          Button(onClick = { viewModel.saveProxySettings() }) {
            Text(stringResource(R.string.label_save))
          }
        }
      }
      // endregion proxy

      Card {
        Column(
            modifier = Modifier.padding(dimensionResource(R.dimen.padding_xl)),
            verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_sm)),
        ) {
          Text(text = stringResource(R.string.label_ssl_check_mode), fontSize = 20.sp)

          RadioGroup(
              options = persistentListOf(SSLCheckMode.CERTIFICATE_ONLY, SSLCheckMode.CERTIFICATE_AND_FINGERPRINT, SSLCheckMode.FINGERPRINT_ONLY),
              selectedOption = viewModel.sslCheckModeDraft,
              onOptionSelected = { viewModel.updateSSLCheckModeDraft(it) },
              label = {
                Text(
                    when (it) {
                      SSLCheckMode.CERTIFICATE_ONLY -> stringResource(R.string.label_ssl_check_mode_certificate_only)
                      SSLCheckMode.CERTIFICATE_AND_FINGERPRINT -> stringResource(R.string.label_ssl_check_mode_certificate_and_fingerprint)
                      SSLCheckMode.FINGERPRINT_ONLY -> stringResource(R.string.label_ssl_check_mode_fingerprint_only)
                      else -> stringResource(R.string.Unknown)
                    },
                )
              },
          )

          Button(onClick = { viewModel.saveSSLCheckModeSettings() }) {
            Text(stringResource(R.string.label_save))
          }
        }
      }

      Row(horizontalArrangement = Arrangement.Center) {
        Text(
            "Oshi Uploader v${BuildConfig.VERSION_NAME}",
            color = MaterialTheme.colorScheme.onSurfaceVariant,
        )
      }

      Spacer(Modifier)
    }
  }
}

@Preview("default")
@Preview("large font", fontScale = 2f)
@Composable
fun SettingsPreview() {
  OshiTheme {
    Settings(
        viewModel = SettingsViewModel(SavedStateHandle(), UserSettingsRepositoryMock()),
        upPress = {},
    )
  }
}
