package org.codeberg.quecomet.oshi.ui.screens.managefile

import androidx.compose.material3.SnackbarDuration
import androidx.compose.runtime.Stable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.flatMapConcat
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.data.UploadedFileRepository
import org.codeberg.quecomet.oshi.exceptions.toUserMessage
import org.codeberg.quecomet.oshi.model.SnackbarMessage
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.ui.navigation.NavArgs
import org.codeberg.quecomet.oshi.utils.WorkState
import org.codeberg.quecomet.oshi.utils.isWorking
import java.net.URLDecoder
import javax.inject.Inject

@HiltViewModel
@Stable
class ManageFileViewModel
@Inject
constructor(
    private val savedStateHandle: SavedStateHandle,
    private val uploadedFileRepository: UploadedFileRepository,
) : ViewModel() {

  val managePath =
      savedStateHandle.getStateFlow<String?>(NavArgs.MANAGE_FILE_PATH_URI_ENCODED, null).map {
        it?.let { URLDecoder.decode(it, "UTF-8") }
      }
  val oshiInstanceId = savedStateHandle.getStateFlow<Int?>(NavArgs.MANAGE_FILE_OSHI_INSTANCE, null)

  @OptIn(ExperimentalCoroutinesApi::class)
  val uploadedFileAndOshiInstance =
      combine(managePath, oshiInstanceId) { mPath, instanceId ->
            uploadedFileRepository.getUploadedFileAndOshiInstanceFlow(
                mPath ?: "non-existing-manage-path", instanceId ?: 0)
          }
          .flatMapConcat { it }
          .stateIn(
              scope = viewModelScope,
              started = SharingStarted.WhileSubscribed(5000),
              initialValue = null,
          )

  var requestFileDeleteWorkState by mutableStateOf<WorkState>(WorkState.NotWorking)
    private set

  var refreshWorkState by mutableStateOf<WorkState>(WorkState.NotWorking)
    private set

  var fileDeleted by mutableStateOf(false)
    private set

  var toggleDestroyWorkState by mutableStateOf<WorkState>(WorkState.NotWorking)
    private set

  var toggleOnionWorkState by mutableStateOf<WorkState>(WorkState.NotWorking)
    private set

  fun requestFileDelete(onSuccess: () -> Unit) {
    if (requestFileDeleteWorkState.isWorking()) return
    requestFileDeleteWorkState = WorkState.Working()
    viewModelScope.launch {
      try {
        val uploadedFile = uploadedFileAndOshiInstance.firstOrNull()?.uploadedFile
        if (uploadedFile == null) {
          requestFileDeleteWorkState = WorkState.NotWorking
          return@launch
        }
        uploadedFileRepository.deleteRemote(uploadedFile.managePath, uploadedFile.oshiInstanceId)
        fileDeleted = true
        uploadedFileRepository.deleteLocal(uploadedFile.managePath, uploadedFile.oshiInstanceId)
        onSuccess()
        requestFileDeleteWorkState = WorkState.NotWorking
      } catch (e: Throwable) {
        requestFileDeleteWorkState = WorkState.Error(e)
        SnackbarManager.showMessage(
            SnackbarMessage.from(
                message = e.toUserMessage(),
                withDismissAction = true,
                duration = SnackbarDuration.Long,
            ),
        )
      }
    }
  }

  fun refreshInfo() {
    if (refreshWorkState.isWorking()) return

    refreshWorkState = WorkState.Working()

    viewModelScope.launch {
      try {
        val uploadedFile = uploadedFileAndOshiInstance.firstOrNull()?.uploadedFile
        if (uploadedFile == null) {
          refreshWorkState = WorkState.NotWorking
          return@launch
        }
        uploadedFileRepository.fetchUploadedFile(
            uploadedFile.managePath, uploadedFile.oshiInstanceId)
        refreshWorkState = WorkState.NotWorking
      } catch (e: Throwable) {
        refreshWorkState = WorkState.Error(e)
        SnackbarManager.showMessage(
            SnackbarMessage.from(
                message = e.toUserMessage(),
                withDismissAction = true,
                duration = SnackbarDuration.Long,
            ))
      }
    }
  }

  fun toggleDestroyAfterDl() {
    if (toggleDestroyWorkState.isWorking()) return
    toggleDestroyWorkState = WorkState.Working()
    viewModelScope.launch {
      try {
        val uploadedFile = uploadedFileAndOshiInstance.firstOrNull()?.uploadedFile
        if (uploadedFile == null) {
          toggleDestroyWorkState = WorkState.NotWorking
          return@launch
        }
        uploadedFileRepository.toggleDestroyAfterDl(
            uploadedFile.managePath, uploadedFile.oshiInstanceId)
        toggleDestroyWorkState = WorkState.NotWorking
      } catch (e: Throwable) {
        toggleDestroyWorkState = WorkState.Error(e)
        SnackbarManager.showMessage(
            SnackbarMessage.from(
                message = e.toUserMessage(),
                withDismissAction = true,
                duration = SnackbarDuration.Long,
            ),
        )
      }
    }
  }

  fun toggleOnionOnly() {
    if (toggleOnionWorkState.isWorking()) return
    toggleOnionWorkState = WorkState.Working()
    viewModelScope.launch {
      try {
        val uploadedFile = uploadedFileAndOshiInstance.firstOrNull()?.uploadedFile
        if (uploadedFile == null) {
          toggleOnionWorkState = WorkState.NotWorking
          return@launch
        }
        uploadedFileRepository.toggleOnionOnly(uploadedFile.managePath, uploadedFile.oshiInstanceId)
        toggleOnionWorkState = WorkState.NotWorking
      } catch (e: Throwable) {
        toggleOnionWorkState = WorkState.Error(e)
        SnackbarManager.showMessage(
            SnackbarMessage.from(
                message = e.toUserMessage(),
                withDismissAction = true,
                duration = SnackbarDuration.Long,
            ),
        )
      }
    }
  }
}
