package org.codeberg.quecomet.oshi.ui.screens.history

import androidx.compose.material3.SnackbarDuration
import androidx.compose.runtime.Stable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import androidx.paging.PagingData
import androidx.paging.cachedIn
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.data.UploadedFileRepository
import org.codeberg.quecomet.oshi.data.room.UploadedFile
import org.codeberg.quecomet.oshi.data.room.UploadedFileAndOshiInstance
import org.codeberg.quecomet.oshi.model.SnackbarMessage
import org.codeberg.quecomet.oshi.model.UserMessage
import org.codeberg.quecomet.oshi.ui.components.SnackbarManager
import org.codeberg.quecomet.oshi.utils.WorkState
import javax.inject.Inject

@HiltViewModel
@Stable
class HistoryViewModel
@Inject
constructor(
    private val savedStateHandle: SavedStateHandle,
    private val uploadedFileRepository: UploadedFileRepository
) : ViewModel() {

  var searchTerm = MutableStateFlow("")
    private set

  @OptIn(ExperimentalCoroutinesApi::class)
  val uploadedFileAndOshiInstancePagingDataFlow: Flow<PagingData<UploadedFileAndOshiInstance>> =
      searchTerm.flatMapLatest {
        uploadedFileRepository.getUploadedFileAndOshiInstanceList(it).cachedIn(viewModelScope)
      }

  var deleteWorkState by mutableStateOf<WorkState>(WorkState.NotWorking)
    private set

  fun deleteUploadedFile(uploadedFile: UploadedFile, onSuccess: () -> Unit) {
    deleteWorkState = WorkState.Working()
    viewModelScope.launch {
      try {
        uploadedFileRepository.deleteRemote(uploadedFile.managePath, uploadedFile.oshiInstanceId)
        uploadedFileRepository.deleteLocal(uploadedFile.managePath, uploadedFile.oshiInstanceId)
        deleteWorkState = WorkState.NotWorking
        onSuccess()
      } catch (e: Throwable) {
        deleteWorkState = WorkState.Error(e)
        SnackbarManager.showMessage(
            SnackbarMessage.from(
                message = UserMessage.from(e.message ?: e.toString()),
                withDismissAction = true,
                duration = SnackbarDuration.Long,
            ))
      }
    }
  }

  fun updateSearchTerm(value: String) {
    searchTerm.value = value
  }
}
