package org.codeberg.quecomet.oshi.ui.navigation

import android.net.Uri
import androidx.annotation.StringRes
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.CloudUpload
import androidx.compose.material.icons.filled.History
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.core.net.toUri
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.navigation.NavBackStackEntry
import androidx.navigation.NavGraphBuilder
import androidx.navigation.NavType
import androidx.navigation.compose.composable
import androidx.navigation.navArgument
import androidx.navigation.navDeepLink
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.ui.screens.history.History
import org.codeberg.quecomet.oshi.ui.screens.history.HistoryViewModel
import org.codeberg.quecomet.oshi.ui.screens.managefile.ManageFile
import org.codeberg.quecomet.oshi.ui.screens.managefile.ManageFileViewModel
import org.codeberg.quecomet.oshi.ui.screens.settings.Settings
import org.codeberg.quecomet.oshi.ui.screens.settings.SettingsViewModel
import org.codeberg.quecomet.oshi.ui.screens.upload.Upload
import org.codeberg.quecomet.oshi.ui.screens.upload.UploadViewModel
import java.net.URLEncoder

const val OSHI_APP_SCHEME = "oshi"

enum class PrimaryDestinations(
    @StringRes val title: Int,
    val icon: ImageVector,
    val route: String
) {
  UPLOAD(R.string.upload, Icons.Default.CloudUpload, "upload"),
  HISTORY(R.string.history, Icons.Default.History, "history"),
}

object SecondaryDestinations {
  const val MANAGE_FILE_ROUTE = "manage_file"
  const val SETTINGS_ROUTE = "settings"
}

object NavArgs {
  const val MANAGE_FILE_PATH_URI_ENCODED = "upload_manage_path"
  const val MANAGE_FILE_OSHI_INSTANCE = "upload_oshi_instance"
}

fun NavGraphBuilder.oshiNavGraph(
    oshiNavController: OshiNavController,
    navigateTo: (String) -> Unit,
    modifier: Modifier = Modifier,
) {
  addUploadGraph(
      oshiNavController = oshiNavController,
      modifier = modifier,
  )
  addHistory(oshiNavController::navigateToManageFile, modifier)
  addManageFile(oshiNavController::upPress, modifier)
  addSettings(oshiNavController::upPress, modifier)
}

private fun NavGraphBuilder.addUploadGraph(
    oshiNavController: OshiNavController,
    modifier: Modifier = Modifier
) {
  composable(PrimaryDestinations.UPLOAD.route) {
    val viewModel = hiltViewModel<UploadViewModel>()
    Upload(
        viewModel = viewModel,
        onNavigateToRoute = oshiNavController::navigateTo,
        modifier = modifier,
    )
  }
}

private fun NavGraphBuilder.addHistory(
    onUploadedFileSelected: (String, Int, NavBackStackEntry) -> Unit,
    modifier: Modifier = Modifier,
) {

  composable(PrimaryDestinations.HISTORY.route) { from ->
    val viewModel = hiltViewModel<HistoryViewModel>()
    History(
        viewModel = viewModel,
        onUploadedFileSelected = { managePath, oshiInstanceId -> onUploadedFileSelected(managePath, oshiInstanceId, from) },
        modifier,
    )
  }
}

private fun NavGraphBuilder.addManageFile(
    upPress: () -> Unit,
    modifier: Modifier = Modifier,
) {
  composable(
      "${SecondaryDestinations.MANAGE_FILE_ROUTE}/{${NavArgs.MANAGE_FILE_OSHI_INSTANCE}}/{${NavArgs.MANAGE_FILE_PATH_URI_ENCODED}}",
      deepLinks =
          listOf(
              navDeepLink {
                uriPattern =
                    "$OSHI_APP_SCHEME://${SecondaryDestinations.MANAGE_FILE_ROUTE}/{${NavArgs.MANAGE_FILE_OSHI_INSTANCE}}/{${NavArgs.MANAGE_FILE_PATH_URI_ENCODED}}"
              },
          ),
      arguments =
          listOf(
              navArgument(NavArgs.MANAGE_FILE_OSHI_INSTANCE) { type = NavType.IntType },
              navArgument(NavArgs.MANAGE_FILE_PATH_URI_ENCODED) { type = NavType.StringType },
          ),
  ) { backStackEntry ->
    val oshiInstanceId = backStackEntry.arguments?.getInt(NavArgs.MANAGE_FILE_OSHI_INSTANCE)
    val managePath = backStackEntry.arguments?.getString(NavArgs.MANAGE_FILE_PATH_URI_ENCODED)
    if (managePath.isNullOrEmpty() || oshiInstanceId == null) {
      upPress()
    } else {
      val viewModel = hiltViewModel<ManageFileViewModel>()
      ManageFile(
          viewModel = viewModel,
          upPress = upPress,
          modifier = modifier,
      )
    }
  }
}

fun getManageFileUri(managePath: String, oshiInstanceId: Int): Uri {
  return "$OSHI_APP_SCHEME://${SecondaryDestinations.MANAGE_FILE_ROUTE}/${oshiInstanceId}/${URLEncoder.encode(managePath, "UTF-8")}"
      .toUri()
}

private fun NavGraphBuilder.addSettings(
    upPress: () -> Unit,
    modifier: Modifier = Modifier,
) {
  composable(SecondaryDestinations.SETTINGS_ROUTE) {
    val viewModel = hiltViewModel<SettingsViewModel>()
    Settings(viewModel, upPress, modifier)
  }
}
