package org.codeberg.quecomet.oshi.ui.navigation

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.widthIn
import androidx.compose.material3.DrawerValue
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalNavigationDrawer
import androidx.compose.material3.PermanentNavigationDrawer
import androidx.compose.material3.rememberDrawerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.currentBackStackEntryAsState
import kotlinx.coroutines.launch
import org.codeberg.quecomet.oshi.DeepLinkEvent
import org.codeberg.quecomet.oshi.DeepLinkHandler

@Composable
fun OshiNavHost(
  navigationType: OshiNavigationType,
  navigationContentPosition: OshiNavigationContentPosition,
  deepLinkHandler: DeepLinkHandler,
) {
  val drawerState = rememberDrawerState(initialValue = DrawerValue.Closed)
  val scope = rememberCoroutineScope()

  val oshiNavController = rememberOshiNavController()
  val navController = oshiNavController.navController

  val navBackStackEntry by navController.currentBackStackEntryAsState()
  val selectedDestination =
      navBackStackEntry?.destination?.route ?: PrimaryDestinations.UPLOAD.route

  val event by deepLinkHandler.deepLinkEvent.collectAsState()

  LaunchedEffect(event) {
    when (val currentEvent = event) {
      is DeepLinkEvent.NavigateWithDeepLink -> navController.handleDeepLink(currentEvent.intent)
      DeepLinkEvent.None -> Unit
    }
    deepLinkHandler.consumeEvent()
  }

  if (navigationType == OshiNavigationType.PERMANENT_NAVIGATION_DRAWER) {
    // TODO check on custom width of PermanentNavigationDrawer: b/232495216
    // TODO check compose Reply app when material is out of alpha for the todo above
    PermanentNavigationDrawer(
        drawerContent = {
          PermanentNavigationDrawerContent(
              selectedDestination = selectedDestination,
              navigationContentPosition = navigationContentPosition,
              navigateTo = oshiNavController::navigateToTopLevel,
          )
        },
    ) {
      OshiAppContent(
          oshiNavController,
          navigationType,
          selectedDestination,
          navigationContentPosition,
          onDrawerClicked = { scope.launch { drawerState.close() } },
      )
    }
  } else {
    ModalNavigationDrawer(
        drawerContent = {
          ModalNavigationDrawerContent(
              selectedDestination = selectedDestination,
              navigationContentPosition = navigationContentPosition,
              navigateTo = oshiNavController::navigateToTopLevel,
              onDrawerClicked = { scope.launch { drawerState.close() } },
          )
        },
        drawerState = drawerState,
    ) {
      OshiAppContent(
          oshiNavController,
          navigationType,
          selectedDestination,
          navigationContentPosition,
          onDrawerClicked = { scope.launch { drawerState.close() } },
      )
    }
  }
}

@Composable
private fun OshiAppContent(
  oshiNavController: OshiNavController,
  navigationType: OshiNavigationType,
  selectedDestination: String,
  navigationContentPosition: OshiNavigationContentPosition,
  modifier: Modifier = Modifier,
  onDrawerClicked: () -> Unit = {}
) {

  Row(modifier = modifier.fillMaxSize()) {
    AnimatedVisibility(visible = navigationType == OshiNavigationType.NAVIGATION_RAIL) {
      OshiNavigationRail(
          selectedDestination = selectedDestination,
          navigationContentPosition = navigationContentPosition,
          navigateTo = oshiNavController::navigateToTopLevel,
          onDrawerClicked = onDrawerClicked,
      )
    }
    Column(
        Modifier
            .fillMaxSize()
            .background(MaterialTheme.colorScheme.background),
    ) {
      Column(
          Modifier
              .weight(1f, true)
              .fillMaxWidth(),
          horizontalAlignment = Alignment.CenterHorizontally,
      ) {
        NavHost(
            navController = oshiNavController.navController,
            startDestination = PrimaryDestinations.UPLOAD.route,
            contentAlignment = Alignment.Center,
            modifier = Modifier
                .widthIn(0.dp, 560.dp)
                .weight(1f, true)
                .fillMaxWidth(),
            enterTransition = { scaleIntoContainer() },
            exitTransition = { scaleOutOfContainer(direction = ScaleTransitionDirection.INWARDS) },
            popEnterTransition = {
              scaleIntoContainer(direction = ScaleTransitionDirection.OUTWARDS)
            },
            popExitTransition = { scaleOutOfContainer() },
        ) {
          oshiNavGraph(
              oshiNavController = oshiNavController,
              navigateTo = oshiNavController::navigateTo,
              modifier = Modifier.weight(1f, true),
          )
        }
      }

      AnimatedVisibility(
          navigationType == OshiNavigationType.BOTTOM_NAVIGATION &&
              PrimaryDestinations.entries.any { selectedDestination.startsWith(it.route) },
      ) {
        OshiBottomNavigationBar(
            selectedDestination = selectedDestination,
            navigateTo = oshiNavController::navigateToTopLevel,
        )
      }
    }
  }
}
