package org.codeberg.quecomet.oshi.ui.components

import androidx.annotation.StringRes
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.WarningAmber
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import org.codeberg.quecomet.oshi.R
import org.codeberg.quecomet.oshi.ui.theme.OshiTheme
import org.codeberg.quecomet.oshi.utils.WorkState
import org.codeberg.quecomet.oshi.utils.isWorking

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ConfirmDeleteDialog(
    modifier: Modifier = Modifier,
    itemLabel: String,
    onConfirm: () -> Unit,
    onDismissRequest: () -> Unit,
    workState: WorkState = WorkState.NotWorking,
    @StringRes confirmMessage: Int = R.string.delete_dialog_confirmation,
) {
  BasicDialog(
      onDismissRequest = { onDismissRequest() },
  ) {
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.spacedBy(dimensionResource(R.dimen.padding_xl)),
        modifier = Modifier.padding(dimensionResource(R.dimen.padding_xl)),
    ) {
      Icon(imageVector = Icons.Default.WarningAmber, contentDescription = null)
      Text(
          text = stringResource(R.string.confirm_dialog_title),
          style = MaterialTheme.typography.titleMedium,
      )
      Text(
          stringResource(confirmMessage, itemLabel),
          style = MaterialTheme.typography.bodyMedium,
      )
      Row(
          verticalAlignment = Alignment.CenterVertically,
          horizontalArrangement = Arrangement.Center,
      ) {
        TextButton(onClick = { onDismissRequest() }, enabled = !workState.isWorking()) {
          Text(stringResource(R.string.cancel))
        }
        Spacer(Modifier.width(10.dp))
        Button(
            onClick = {
              if (!workState.isWorking()) {
                onConfirm()
              }
            },
            colors =
                ButtonDefaults.buttonColors(
                    containerColor = MaterialTheme.colorScheme.errorContainer,
                    contentColor = MaterialTheme.colorScheme.onErrorContainer,
                ),
        ) {
          if (workState.isWorking()) {
            CircularProgressIndicator(
                modifier.size(18.dp), color = MaterialTheme.colorScheme.onErrorContainer)
          } else {
            Text(stringResource(R.string.delete))
          }
        }
      }
    }
  }
}

@Preview("Confirm Delete")
@Composable
fun ConfirmDeleteDialogPreview() {
  OshiTheme {
    Surface { ConfirmDeleteDialog(onConfirm = {}, onDismissRequest = {}, itemLabel = "item name") }
  }
}
