package org.codeberg.quecomet.oshi.data.room

import androidx.paging.PagingSource
import androidx.room.Dao
import androidx.room.Query
import androidx.room.Transaction
import androidx.room.Update
import kotlinx.coroutines.flow.Flow

/** [Room] DAO for [UploadedFile] related operations. */
@Dao
abstract class UploadedFileDao : BaseDao<UploadedFile> {

  @Transaction
  @Query("SELECT * FROM `UploadedFile` WHERE managePath = :managePath AND oshiInstanceId = :oshiInstanceId")
  abstract suspend fun uploadedFileAndOshiInstanceByManagePath(managePath: String, oshiInstanceId: Int): UploadedFileAndOshiInstance?

  @Transaction
  @Query("SELECT * FROM `UploadedFile` WHERE managePath = :managePath AND oshiInstanceId = :oshiInstanceId")
  abstract fun uploadedFileAndOshiInstanceByManagePathFlow(managePath: String, oshiInstanceId: Int): Flow<UploadedFileAndOshiInstance?>

  @Transaction
  @Query("SELECT * FROM UploadedFile ORDER BY createdAt DESC")
  abstract fun uploadedFileAndOshiInstanceSortedByCreatedAtPagingSource(): PagingSource<Int, UploadedFileAndOshiInstance>

  @Transaction
  @Query("SELECT UploadedFile.* FROM UploadedFile join OshiInstance WHERE oshiInstanceId = OshiInstance.id AND (filename LIKE '%' || :searchTerm || '%' or originalFilename LIKE '%' || :searchTerm || '%' or host LIKE '%' || :searchTerm || '%') GROUP by managePath ORDER BY createdAt DESC")
  abstract fun uploadedFileAndOshiInstanceSortedByCreatedAtPagingSourceFiltered(searchTerm: String): PagingSource<Int, UploadedFileAndOshiInstance>

  @Query("SELECT COUNT(*) FROM UploadedFile")
  abstract suspend fun count(): Int

  @Query("SELECT EXISTS(SELECT 1 FROM `UploadedFile` WHERE managePath = :managePath AND oshiInstanceId = :oshiInstanceId LIMIT 1)")
  abstract suspend fun exists(managePath: String, oshiInstanceId: Int): Boolean

  @Update(entity = UploadedFile::class)
  abstract suspend fun setDeletedInRemote(entity: UploadedFileWithDeletedInRemote)

  @Update(entity = UploadedFile::class)
  abstract suspend fun setIsOnionOnly(entity: UploadedFileWithIsOnionOnly)

  @Update(entity = UploadedFile::class)
  abstract suspend fun setDestroyAfterDl(entity: UploadedFileWithDestroyAfterDl)

  @Query("DELETE FROM `UploadedFile` WHERE managePath = :managePath AND oshiInstanceId = :oshiInstanceId")
  abstract suspend fun delete(managePath: String, oshiInstanceId: Int)
}
