package org.codeberg.quecomet.oshi.data.room

import androidx.compose.runtime.Immutable
import androidx.room.ColumnInfo
import androidx.room.Embedded
import androidx.room.Entity
import androidx.room.ForeignKey
import androidx.room.Index
import androidx.room.Relation
import java.util.Date

@Immutable
@Entity(
    primaryKeys = ["managePath", "oshiInstanceId"],
    indices =
        [
            Index("createdAt"),
            Index("oshiInstanceId"),
        ],
    foreignKeys =
        [
            ForeignKey(
                entity = OshiInstance::class,
                childColumns = ["oshiInstanceId"],
                parentColumns = ["id"],
                onDelete = ForeignKey.CASCADE,
                onUpdate = ForeignKey.CASCADE,
            )])
data class UploadedFile(
    val managePath: String,
    val oshiInstanceId: Int,
    val downloadPath: String,
    val filename: String,
    val originalFilename: String = filename,
    val size: Long,
    val mimeType: String = "",
    @ColumnInfo(defaultValue = "0") val destroyAfterDl: Boolean,
    @ColumnInfo(defaultValue = "0") val isOnionOnly: Boolean = false,
    @ColumnInfo(defaultValue = CURRENT_TIMESTAMP_EXPRESSION)
    val expiresAt: Date = Date(Date().time + (1440 * 60 * 1000)),
    @ColumnInfo(defaultValue = CURRENT_TIMESTAMP_EXPRESSION) val createdAt: Date = Date(),
    @ColumnInfo(defaultValue = "0") val deletedInRemote: Boolean = false,
    @ColumnInfo(defaultValue = "null") val hits: String? = null,
    @ColumnInfo(defaultValue = "null") val hashsumSha1: String? = null,
) {
  override fun toString(): String {
    return originalFilename
  }
}

@Immutable
data class UploadedFileWithDestroyAfterDl(
    val managePath: String,
    val oshiInstanceId: Int,
    val destroyAfterDl: Boolean,
)

@Immutable
data class UploadedFileWithIsOnionOnly(
    val managePath: String,
    val oshiInstanceId: Int,
    val isOnionOnly: Boolean,
)

@Immutable
data class UploadedFileWithDeletedInRemote(
    val managePath: String,
    val oshiInstanceId: Int,
    val deletedInRemote: Boolean,
)

@Immutable
data class UploadedFileAndOshiInstance(
    @Embedded val uploadedFile: UploadedFile,
    @Relation(parentColumn = "oshiInstanceId", entityColumn = "id") val oshiInstance: OshiInstance
)
